/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.agent;

import com.sshtools.j2ssh.SshException;

import com.sshtools.j2ssh.configuration.Authorization;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.ServerConfiguration;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.platform.NativeAuthenticationProvider;

import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.TransportProtocolException;
import com.sshtools.j2ssh.transport.SshMessage;

import com.sshtools.j2ssh.transport.publickey.InvalidSshKeyException;
import com.sshtools.j2ssh.transport.publickey.InvalidSshKeySignatureException;
import com.sshtools.j2ssh.transport.publickey.SECSHPublicKeyFormat;
import com.sshtools.j2ssh.transport.publickey.SshKeyPair;
import com.sshtools.j2ssh.transport.publickey.SshKeyPairFactory;
import com.sshtools.j2ssh.transport.publickey.SshPrivateKey;
import com.sshtools.j2ssh.transport.publickey.SshPrivateKeyFile;
import com.sshtools.j2ssh.transport.publickey.SshPublicKey;
import com.sshtools.j2ssh.transport.publickey.SshPublicKeyFile;
import com.sshtools.j2ssh.transport.publickey.SshPublicKeyFormatFactory;
import com.sshtools.j2ssh.transport.publickey.SshtoolsPrivateKeyFormat;

import org.apache.log4j.Logger;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Window;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import com.sshtools.j2ssh.authentication.*;

public class AgentAuthenticationClient extends SshAuthenticationClient {
    private static Logger log = Logger.getLogger(PublicKeyAuthenticationClient.class);
    protected SshPublicKey key;
    protected SshAgentClient agent;

    public AgentAuthenticationClient() {

    }

    public void setKey(SshPublicKey key) {
        this.key = key;
    }

    public void setAgent(SshAgentClient agent) {
      this.agent = agent;
    }

    public String getMethodName() {
        return "publickey";
    }

    public boolean acceptsKey(AuthenticationProtocolClient authentication,
                                    String username,
                                    String serviceToStart,
                                    SshPublicKey key) throws IOException {

        authentication.registerMessage(SshMsgUserAuthPKOK.class,
                                       SshMsgUserAuthPKOK.SSH_MSG_USERAUTH_PK_OK);

        log.info("Determining if server can accept public key for authentication");

        ByteArrayWriter baw = new ByteArrayWriter();

        // Now prepare and send the message
        baw.write(0);
        baw.writeString(key.getAlgorithmName());
        baw.writeBinaryString(key.getEncoded());

        SshMessage msg = new SshMsgUserAuthRequest(username,
                serviceToStart, getMethodName(), baw.toByteArray());

        authentication.sendMessage(msg);

        try {
          msg = authentication.readMessage(SshMsgUserAuthPKOK.
                                           SSH_MSG_USERAUTH_PK_OK);

          if(msg instanceof SshMsgUserAuthPKOK)
            return true;
          else
            throw new IOException("Unexpected message returned from readMessage");
        } catch(TerminatedStateException ex) {
          return false;
        }



    }

    public void authenticate(AuthenticationProtocolClient authentication,
        String serviceToStart) throws IOException, TerminatedStateException {
        if ((getUsername() == null) || (key == null) || (agent == null)) {
            throw new AuthenticationProtocolException(
                "You must supply a username, key and agent");
        }

        ByteArrayWriter baw = new ByteArrayWriter();

        log.info("Generating data to sign");

        log.info("Preparing public key authentication request");

        // Now prepare and send the message
        baw.write(1);
        baw.writeString(key.getAlgorithmName());
        baw.writeBinaryString(key.getEncoded());

        // Create the signature data
        ByteArrayWriter data = new ByteArrayWriter();
        data.writeBinaryString(authentication.getSessionIdentifier());
        data.write(SshMsgUserAuthRequest.SSH_MSG_USERAUTH_REQUEST);
        data.writeString(getUsername());
        data.writeString(serviceToStart);
        data.writeString(getMethodName());
        data.write(1);
        data.writeString(key.getAlgorithmName());
        data.writeBinaryString(key.getEncoded());


        // Generate the signature
        baw.writeBinaryString(agent.hashAndSign(key, data.toByteArray()));

        SshMsgUserAuthRequest msg = new SshMsgUserAuthRequest(getUsername(),
                serviceToStart, getMethodName(), baw.toByteArray());

        authentication.sendMessage(msg);
    }

    public boolean showAuthenticationDialog(Component parent) {
        return false;
    }

    public Properties getPersistableProperties() {
        Properties properties = new Properties();

        return properties;
    }

    public void setPersistableProperties(Properties properties) {

    }

    public boolean canAuthenticate() {
        return ((getUsername() != null) && (key != null) && (agent!=null));
    }
}
