package com.sshtools.j2ssh.agent;


import com.sshtools.j2ssh.io.*;
import java.io.*;

public class KeyConstraints {

  public static final long NO_TIMEOUT = 0;
  public static final long NO_LIMIT = 0xffffffffL;


  protected static final int SSH_AGENT_CONSTRAINT_TIMEOUT = 50;
  protected static final int SSH_AGENT_CONSTRAINT_USE_LIMIT = 51;
  protected static final int SSH_AGENT_CONSTRAINT_FORWARDING_STEPS = 52;
  protected static final int SSH_AGENT_CONSTRAINT_FORWARDING_PATH = 100;
  protected static final int SSH_AGENT_CONSTRAINT_SSH1_COMPAT = 150;
  protected static final int SSH_AGENT_CONSTRAINT_NEED_USER_VERIFICATION = 151;

  private UnsignedInteger32 timeout = new UnsignedInteger32(NO_TIMEOUT);
  private UnsignedInteger32 uselimit = new UnsignedInteger32(NO_LIMIT);
  private UnsignedInteger32 maxsteps = new UnsignedInteger32(NO_LIMIT);
  private String forwardingpath = "";
  private boolean userverify = false;
  private boolean compat = false;

  private long keyadded = System.currentTimeMillis();
  private long usedcount = 0;

  public KeyConstraints() {
  }

  public KeyConstraints(ByteArrayReader bar) throws IOException {
    while(bar.available() > 0) {
      switch(bar.read() & 0xFF) {
        case SSH_AGENT_CONSTRAINT_TIMEOUT:
          timeout = bar.readUINT32();
          break;
        case SSH_AGENT_CONSTRAINT_USE_LIMIT:
          uselimit = bar.readUINT32();
          break;
        case SSH_AGENT_CONSTRAINT_FORWARDING_STEPS:
          maxsteps = bar.readUINT32();
          break;
        case SSH_AGENT_CONSTRAINT_FORWARDING_PATH:
          forwardingpath = bar.readString();
          break;
        case SSH_AGENT_CONSTRAINT_SSH1_COMPAT:
          compat = (bar.read()!=0);
          break;
        case SSH_AGENT_CONSTRAINT_NEED_USER_VERIFICATION:
          userverify = (bar.read()!=0);
          break;
      }
    }
  }

  public void setKeyTimeout(UnsignedInteger32 timeout) {
    this.timeout = timeout;
  }

  public void setKeyUseLimit(int uselimit) {
    this.uselimit = new UnsignedInteger32(uselimit);
  }

  public void setMaximumForwardingSteps(int maxsteps) {
    this.maxsteps = new UnsignedInteger32(maxsteps);
  }

  public void setForwardingPath(String forwardingpath) {
    this.forwardingpath = forwardingpath;
  }

  public void setRequiresUserVerification(boolean userverify) {
    this.userverify = userverify;
  }

  public void setSSH1Compatible(boolean compat) {
    this.compat = compat;
  }

  public long getKeyTimeout() {
    return timeout.longValue();
  }

  public long getKeyUseLimit() {
    return uselimit.longValue();
  }

  public long getMaximumForwardingSteps() {
    return maxsteps.longValue();
  }

  public long getUsedCount() {
    return usedcount;
  }

  public boolean hasTimedOut() {
    return (timeout.longValue() != 0) ?
        ((System.currentTimeMillis()-keyadded)/1000) > timeout.longValue()
        : false;
  }

  public boolean canUse() {
    return (uselimit.longValue() != 0) ? (usedcount < uselimit.longValue()) : true;
  }

  public void use() {
    usedcount++;
  }

  public String getForwardingPath() {
    return forwardingpath;
  }

  public boolean requiresUserVerification() {
    return userverify;
  }

  public boolean isSSH1Compatible() {
    return compat;
  }


  public byte[] toByteArray() throws IOException {
    ByteArrayWriter baw = new ByteArrayWriter();
    baw.write(SSH_AGENT_CONSTRAINT_TIMEOUT);
    baw.writeUINT32(timeout);

    baw.write(SSH_AGENT_CONSTRAINT_USE_LIMIT);
    baw.writeUINT32(uselimit);

    baw.write(SSH_AGENT_CONSTRAINT_FORWARDING_STEPS);
    baw.writeUINT32(maxsteps);

    baw.write(SSH_AGENT_CONSTRAINT_FORWARDING_PATH);
    baw.writeString(forwardingpath);

    baw.write(SSH_AGENT_CONSTRAINT_SSH1_COMPAT);
    baw.write(compat?0:1);

    baw.write(SSH_AGENT_CONSTRAINT_NEED_USER_VERIFICATION);
    baw.write(userverify?0:1);

    return baw.toByteArray();
  }

}