/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.authentication;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Iterator;

import org.apache.log4j.Logger;
import com.sshtools.j2ssh.configuration.Authorization;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.ServerConfiguration;
import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;
import com.sshtools.j2ssh.platform.NativeAuthenticationProvider;
import com.sshtools.j2ssh.transport.publickey.SshKeyPair;
import com.sshtools.j2ssh.transport.publickey.SshKeyPairFactory;
import com.sshtools.j2ssh.transport.publickey.SshPublicKey;
import com.sshtools.j2ssh.transport.publickey.SshPublicKeyFile;

public class PublicKeyAuthenticationServer
    extends SshAuthenticationServer {
  private Logger log = Logger.getLogger(PublicKeyAuthenticationServer.class);

  private static Class pkv = AuthorizationFileVerification.class;

  public PublicKeyAuthenticationServer() {
  }

  public String getMethodName() {
    return "publickey";
  }

  public static void setVerificationImpl(Class pkv) {
    PublicKeyAuthenticationServer.pkv = pkv;
  }

  public int authenticate(AuthenticationProtocolServer authentication,
                          SshMsgUserAuthRequest msg) throws IOException { //, Map nativeSettings)

    ByteArrayReader bar = new ByteArrayReader(msg.getRequestData());

    // If check == 0 then authenticate, otherwise just inform that
    // the authentication can continue with the key supplied
    int check = bar.read();
    String algorithm = bar.readString();
    byte[] encoded = bar.readBinaryString();
    byte[] signature = null;

    if (check == 0) {
      // Verify that the public key can be used for authenticaiton
      boolean ok = SshKeyPairFactory.supportsKey(algorithm);

      // Send the reply
      SshMsgUserAuthPKOK reply = new SshMsgUserAuthPKOK(ok, algorithm,
          encoded);
      authentication.sendMessage(reply);

      return AuthenticationProtocolState.READY;
    }
    else {

      signature = bar.readBinaryString();

      NativeAuthenticationProvider authProv = NativeAuthenticationProvider
          .getInstance();

      if (authProv == null) {
        log.error(
            "Authentication failed because no native authentication provider is available");

        return AuthenticationProtocolState.FAILED;
      }

      if (!authProv.logonUser(msg.getUsername())) { //, nativeSettings)) {
        log.info("Authentication failed because " + msg.getUsername()
                 + " is not a valid username");

        return AuthenticationProtocolState.FAILED;
      }

      /**
       * This part could be extracted to allow for other key verification policies
       * such as Open3SP
       */
      try {
        PublicKeyVerification verify = (PublicKeyVerification) pkv.newInstance();
        if (verify.verifyKeySignature(msg.getUsername(), algorithm, encoded,
                                      msg.getServiceName(),
                                      authentication.getSessionIdentifier(),
                                      signature)) {
          return AuthenticationProtocolState.COMPLETE;
        }

      }
      catch (Exception ex) {
        log.error("Failed to create an instance of the verification implementation", ex);

      }




    }

    return AuthenticationProtocolState.FAILED;
  }
}
