/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.authentication;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;
import com.sshtools.j2ssh.transport.InvalidMessageException;
import com.sshtools.j2ssh.transport.SshMessage;

public class SshMsgUserAuthFailure
    extends SshMessage {
  protected final static int SSH_MSG_USERAUTH_FAILURE = 51;
  private List auths;
  private boolean partialSuccess;

  public SshMsgUserAuthFailure() {
    super(SSH_MSG_USERAUTH_FAILURE);
  }

  public SshMsgUserAuthFailure(String auths, boolean partialSuccess) throws
      InvalidMessageException {
    super(SSH_MSG_USERAUTH_FAILURE);

    loadListFromDelimString(auths);

    this.partialSuccess = partialSuccess;
  }

  public List getAvailableAuthentications() {
    return auths;
  }

  public String getMessageName() {
    return "SSH_MSG_USERAUTH_FAILURE";
  }

  public boolean getPartialSuccess() {
    return partialSuccess;
  }

  protected void constructByteArray(ByteArrayWriter baw) throws
      InvalidMessageException {
    try {
      String authMethods = null;
      Iterator it = auths.iterator();

      while (it.hasNext()) {
        authMethods = ( (authMethods == null) ? "" : (authMethods + ","))
            + (String) it.next();
      }

      baw.writeString(authMethods);

      baw.write( (partialSuccess ? 1 : 0));
    }
    catch (IOException ioe) {
      throw new InvalidMessageException("Invalid message data");
    }
  }

  protected void constructMessage(ByteArrayReader bar) throws
      InvalidMessageException {
    try {
      String auths = bar.readString();
      partialSuccess = ( (bar.read() != 0) ? true : false);

      loadListFromDelimString(auths);
    }
    catch (IOException ioe) {
      throw new InvalidMessageException("Invalid message data");
    }
  }

  private void loadListFromDelimString(String list) {
    StringTokenizer tok = new StringTokenizer(list, ",");

    auths = new ArrayList();

    while (tok.hasMoreElements()) {
      auths.add(tok.nextElement());
    }
  }
}
