/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.configuration;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.xml.sax.helpers.DefaultHandler;

import java.io.IOException;
import java.io.InputStream;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;


public class Authorization extends DefaultHandler {
    private static final String AUTHORIZEDKEYS_ELEMENT = "AuthorizedKeys";
    private static final String KEY_ELEMENT = "Key";
    private ArrayList authorizedKeys = new ArrayList();

    public Authorization(InputStream in)
        throws SAXException, ParserConfigurationException, IOException {
        SAXParserFactory saxFactory = SAXParserFactory.newInstance();
        SAXParser saxParser = saxFactory.newSAXParser();
        authorizedKeys.clear();
        saxParser.parse(in, new AuthorizedKeysSAXHandler());
    }

    public Authorization() {
        // Creates an empty authorization file
    }

    public List getAuthorizedKeys() {
        return (List) authorizedKeys.clone();
    }

    public void addKey(String keyfile) {
        authorizedKeys.add(keyfile);
    }

    public String toString() {
        String xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
        xml += "<!-- SSHTools Public Key Authorization File -->\n";
        xml += ("<" + AUTHORIZEDKEYS_ELEMENT + ">\n");
        xml += "<!-- Enter authorized public key elements here -->\n";

        Iterator it = authorizedKeys.iterator();

        while (it.hasNext()) {
            xml += ("   <" + KEY_ELEMENT + ">" + it.next().toString() + "</"
            + KEY_ELEMENT + ">\n");
        }

        xml += ("</" + AUTHORIZEDKEYS_ELEMENT + ">");

        return xml;
    }

    class AuthorizedKeysSAXHandler extends DefaultHandler {
        private String currentElement = null;

        public void startElement(String uri, String localName, String qname,
            Attributes attrs) throws SAXException {
            if (currentElement == null) {
                if (!qname.equals(AUTHORIZEDKEYS_ELEMENT)) {
                    throw new SAXException("Unexpected root element " + qname);
                }
            } else {
                if (currentElement.equals(AUTHORIZEDKEYS_ELEMENT)) {
                    if (!qname.equals(KEY_ELEMENT)) {
                        throw new SAXException("Unexpected element " + qname);
                    }
                } else {
                    throw new SAXException("Unexpected element " + qname);
                }
            }

            currentElement = qname;
        }

        public void characters(char[] ch, int start, int length)
            throws SAXException {
            if (currentElement != null) {
                if (currentElement.equals(KEY_ELEMENT)) {
                    String key = new String(ch, start, length);
                    authorizedKeys.add(key);
                }
            }
        }

        public void endElement(String uri, String localName, String qname)
            throws SAXException {
            if (currentElement != null) {
                if (!currentElement.equals(qname)) {
                    throw new SAXException("Unexpected end element found "
                        + qname);
                }

                if (currentElement.equals(KEY_ELEMENT)) {
                    currentElement = AUTHORIZEDKEYS_ELEMENT;
                } else if (currentElement.equals(AUTHORIZEDKEYS_ELEMENT)) {
                    currentElement = null;
                } else {
                    throw new SAXException("Unexpected end element " + qname);
                }
            }
        }
    }
}
