/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.configuration.automate;

import com.sshtools.j2ssh.transport.publickey.InvalidSshKeyException;
import com.sshtools.j2ssh.transport.publickey.SECSHPublicKeyFormat;
import com.sshtools.j2ssh.transport.publickey.SshPublicKey;
import com.sshtools.j2ssh.transport.publickey.SshPublicKeyFile;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import java.util.Iterator;
import java.util.Map;


public class SSH2AuthorizedKeysFormat implements AuthorizedKeysFormat {
    private static final String header = "# Open3SP auto-generated authorization file\n";
    private static final String key = "key ";

    public byte[] format(AuthorizedKeys keys)
        throws IOException, InvalidSshKeyException {
        throw new java.lang.UnsupportedOperationException(
            "SSH2 authorized keys format requries seperate key files!");
    }

    public AuthorizedKeys unformat(byte[] formatted)
        throws IOException, InvalidSshKeyException {
        throw new java.lang.UnsupportedOperationException(
            "SSH2 authorized keys format requries seperate key files!");
    }

    public byte[] format(AuthorizedKeys keys, AuthorizedKeysFileSaver saver)
        throws IOException, InvalidSshKeyException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        out.write(header.getBytes());

        SshPublicKeyFile pubfile;
        SECSHPublicKeyFormat secsh = new SECSHPublicKeyFormat();
        Map.Entry entry;

        for (Iterator it = keys.getAuthorizedKeys().entrySet().iterator();
                (it != null) && it.hasNext();) {
            entry = (Map.Entry) it.next();

            // Write out the public key file
            String username = (String) entry.getKey();
            String filename = username + ".pub";
            secsh.setComment(username);
            pubfile = SshPublicKeyFile.create((SshPublicKey) entry.getValue(),
                    secsh);
            saver.saveFile(filename, pubfile.toString().getBytes());

            // Write out the key entry
            out.write(key.getBytes());
            out.write(filename.getBytes());
            out.write('\n');
        }

        return out.toByteArray();
    }

    public AuthorizedKeys unformat(byte[] formatted,
        AuthorizedKeysFileLoader loader)
        throws IOException, InvalidSshKeyException {
        AuthorizedKeys keys = new AuthorizedKeys();
        BufferedReader reader = new BufferedReader(new InputStreamReader(
                    new ByteArrayInputStream(formatted)));
        String line;
        SshPublicKeyFile pubfile;
        String filename;
        String username;

        while ((line = reader.readLine()) != null) {
            if (line.trim().startsWith("key")) {
                // Get the filename and load
                filename = line.substring(line.trim().lastIndexOf(" ") + 1)
                               .trim();
                pubfile = SshPublicKeyFile.parse(loader.loadFile(filename));

                // Get the username from the filename - .pub
                username = filename.substring(0, filename.length() - 4);
                keys.addKey(username, pubfile.toPublicKey());
            }
        }

        return keys;
    }

    public boolean requiresKeyFiles() {
        return true;
    }
}
