/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.forwarding;

import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;

import com.sshtools.j2ssh.connection.ChannelState;
import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelEventListener;
import com.sshtools.j2ssh.connection.ConnectionProtocol;

import com.sshtools.j2ssh.io.IOStreamConnectorListener;

import com.sshtools.j2ssh.util.InvalidStateException;
import com.sshtools.j2ssh.util.StartStopState;

import org.apache.log4j.Logger;

import java.net.Socket;
import java.net.SocketAddress;
import java.net.InetSocketAddress;

import java.util.List;
import java.util.Vector;

import java.io.IOException;

import javax.swing.event.EventListenerList;


public class ForwardingConfiguration {

    private static Logger log = Logger.getLogger(ForwardingConfiguration.class);
    protected StartStopState state = new StartStopState(StartStopState.STOPPED);
    protected String addressToBind;
    protected String hostToConnect;
    protected String name;
    protected int portToBind;
    protected int portToConnect;

    protected ForwardingConfigurationMonitor monitor
        = new ForwardingConfigurationMonitor();

    protected EventListenerList listenerList = new EventListenerList();
    private List activeForwardings = new Vector();


    public ForwardingConfiguration(String name, String addressToBind,
        int portToBind, String hostToConnect, int portToConnect) {

        this.addressToBind = addressToBind;
        this.portToBind = portToBind;
        this.name = name;
        this.hostToConnect = hostToConnect;
        this.portToConnect = portToConnect;
    }


    protected ForwardingConfiguration(String addressToBind, int portToBind) {
        this(addressToBind + ":" + String.valueOf(portToBind), addressToBind,
            portToBind, "[Specified by connecting computer]", -1);
    }


    public void addForwardingConfigurationListener(
        ForwardingConfigurationListener l) {
      listenerList.add(ForwardingConfigurationListener.class, l);
    }

    public void removeForwardingConfigurationListener(
        ForwardingConfigurationListener l) {
      listenerList.remove(ForwardingConfigurationListener.class, l);
    }

    public List getActiveForwardingChannels() {
      return activeForwardings;
    }


    public String getAddressToBind() {
        return addressToBind;
    }

    public String getHostToConnect() {
        return hostToConnect;
    }

    public String getName() {
        return name;
    }

    public int getPortToBind() {
        return portToBind;
    }

    public int getPortToConnect() {
        return portToConnect;
    }

    public StartStopState getState() {
        return state;
    }

    public void start() throws IOException {
        state.setValue(StartStopState.STARTED);
    }

    public void stop() {
        state.setValue(StartStopState.STOPPED);
    }

    public ForwardingChannel createForwardingChannel(String type,
            String hostToConnect, int portToConnect, String originatingHost, int originatingPort)
            throws ForwardingConfigurationException {
            if (state.getValue() == StartStopState.STOPPED) {
                throw new ForwardingConfigurationException(
                    "The forwarding has been stopped");
            }

            if (!type.equals(ForwardingChannel.LOCAL_FORWARDING_CHANNEL)
                    && !type.equals(ForwardingChannel.REMOTE_FORWARDING_CHANNEL)
                    && !type.equals(ForwardingChannel.X11_FORWARDING_CHANNEL)) {
                throw new ForwardingConfigurationException(
                    "The channel type must either be "
                    + "ForwardingChannel.LOCAL_FORWARDING_CHANNEL_TYPE or "
                    + "ForwardingChannel.REMOTE_FORWARDING_CHANNEL_TYPE");
            }

            ForwardingChannel channel = new ForwardingChannel(type,
                this,
                originatingHost,
                originatingPort);

            channel.addEventListener(monitor);

            return channel;

      }

      public class ForwardingConfigurationMonitor implements ChannelEventListener {

            public void onChannelOpen(Channel channel) {

              if (log.isDebugEnabled()) {
                ForwardingChannel fch = (ForwardingChannel) channel;
                log.debug("Opening forwarding channel from "
                          + fch.getRemoteSocketAddress().getHostName()
                          + ":"
                          + String.valueOf(fch.getRemoteSocketAddress().getPort()));
              }

              // Add channel to the active forwardings
              activeForwardings.add(channel);

              ForwardingConfigurationListener[] l = (ForwardingConfigurationListener[])
                  listenerList
                  .getListeners(ForwardingConfigurationListener.class);

              for (int i = (l.length - 1); i >= 0; i--)
                l[i].opened((ForwardingChannel)channel);

            }

            public void onChannelEOF(Channel channel) {
              // Close the OutputStream to force the channel to close
              try {
                channel.getOutputStream().close();
              }
              catch (IOException ex) {
              }
            }

            public void onChannelClose(Channel channel) {

              if(log.isDebugEnabled()) {
                ForwardingChannel fch = (ForwardingChannel)channel;
                log.debug("Closing forwarding channel from "
                          + fch.getRemoteSocketAddress().getHostName()
                          + ":"
                          + String.valueOf(fch.getRemoteSocketAddress().getPort()));
              }

              // Remove channel from the active forwardings
              activeForwardings.remove(channel);

              ForwardingConfigurationListener[] l = (ForwardingConfigurationListener[])
                  listenerList
                  .getListeners(ForwardingConfigurationListener.class);

              for (int i = (l.length - 1); i >= 0; i--)
                l[i].closed( (ForwardingChannel) channel);

            }

            public void onDataReceived(Channel channel, int numBytes) {
              ForwardingConfigurationListener[] l = (ForwardingConfigurationListener[])
                  listenerList
                  .getListeners(ForwardingConfigurationListener.class);

              for (int i = (l.length - 1); i >= 0; i--)
                l[i].dataReceived((ForwardingChannel)channel, numBytes);

            }

            public void onDataSent(Channel channel, int numBytes) {
              ForwardingConfigurationListener[] l = (ForwardingConfigurationListener[]) listenerList
                                          .getListeners(ForwardingConfigurationListener.class);
              for (int i = (l.length - 1); i >= 0; i--)
                  l[i].dataSent((ForwardingChannel)channel, numBytes);

            }

       }

}
