/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;


public class DynamicBuffer {
    protected static final int DEFAULT_BUFFER_SIZE = 32768;
    protected byte[] buf;
    protected int writepos = 0;
    protected int readpos = 0;
    private Object lock = new Object();
    protected InputStream in;
    protected OutputStream out;

    public DynamicBuffer() {
        buf = new byte[DEFAULT_BUFFER_SIZE];
        in = new DynamicBufferInputStream();
        out = new DynamicBufferOutputStream();
    }

    public InputStream getInputStream() {
        return in;
    }

    public OutputStream getOutputStream() {
        return out;
    }

    private synchronized void verifyBufferSize(int count) {
        synchronized (lock) {
            // If there is not enough data in the buffer, then first attempt to
            // move the unread data back to the beginning
            if (count > (buf.length - writepos)) {
                System.arraycopy(buf, readpos, buf, 0, writepos - readpos);
                writepos -= readpos;
                readpos = 0;
            }

            // Now double check and increase the buffer size if necersary
            if (count > (buf.length - writepos)) {
                byte[] tmp = new byte[buf.length + DEFAULT_BUFFER_SIZE];
                System.arraycopy(buf, 0, tmp, 0, writepos - readpos);
                buf = tmp;
            }
        }
    }

    private synchronized int available() {
        synchronized (lock) {
            return writepos - readpos;
        }
    }

    private synchronized void block() {
        // Block and wait for more data
        while (readpos >= writepos) {
            try {
                wait();
            } catch (InterruptedException ex) {
            }
        }
    }

    public synchronized void write(int b) {
        synchronized (lock) {
            verifyBufferSize(1);
            buf[writepos] = (byte) b;
            writepos++;
        }

        notifyAll();
    }

    public synchronized void write(byte[] data, int offset, int len)
        throws IOException {
        synchronized (lock) {
            verifyBufferSize(len);
            System.arraycopy(data, offset, buf, writepos, len);
            writepos += len;
        }

        notifyAll();
    }

    public synchronized int read() throws IOException {
        block();

        synchronized (lock) {
            return (int) buf[readpos++];
        }
    }

    public synchronized int read(byte[] data, int offset, int len)
        throws IOException {
        block();

        synchronized (lock) {
            int read = (len > (writepos - readpos)) ? (writepos - readpos) : len;
            System.arraycopy(buf, readpos, data, offset, read);
            readpos += read;

            return read;
        }
    }

    public synchronized void flush() throws IOException {
        notifyAll();
    }

    class DynamicBufferInputStream extends InputStream {
        public int read() throws IOException {
            return DynamicBuffer.this.read();
        }

        public int read(byte[] data, int offset, int len)
            throws IOException {
            return DynamicBuffer.this.read(data, offset, len);
        }

        public int available() {
            return DynamicBuffer.this.available();
        }
    }

    class DynamicBufferOutputStream extends OutputStream {
        public void write(int b) {
            DynamicBuffer.this.write(b);
        }

        public void write(byte[] data, int offset, int len)
            throws IOException {
            DynamicBuffer.this.write(data, offset, len);
        }

        public void flush() throws IOException {
            DynamicBuffer.this.flush();
        }
    }
}
