/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.sftp;

import com.sshtools.j2ssh.SshRuntimeException;
import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.*;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.PlatformConfiguration;
import com.sshtools.j2ssh.configuration.VFSMount;
import com.sshtools.j2ssh.configuration.VFSPermission;

import com.sshtools.j2ssh.io.UnsignedInteger32;
import com.sshtools.j2ssh.io.UnsignedInteger64;

import com.sshtools.j2ssh.platform.*;

import com.sshtools.j2ssh.sftp.FileAttributes;

import org.apache.log4j.Logger;

import java.io.*;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.RandomAccessFile;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


public class VirtualFileSystem extends NativeFileSystemProvider {
    private static String USER_HOME = "/home/";
    private static Map vfsmounts;
    private static VFSMount vfsroot;
    private static Logger log = Logger.getLogger(VirtualFileSystem.class);

    static {
        try {
            vfsmounts = ConfigurationLoader.getPlatformConfiguration()
                                           .getVFSMounts();
            vfsroot = ConfigurationLoader.getPlatformConfiguration().getVFSRoot();
        } catch (ConfigurationException ex) {
            log.error("Failed to initialize the Virtual File System", ex);
        }
    }

    private Map openFiles = new HashMap();

    public VirtualFileSystem() throws IOException {
        if (!ConfigurationLoader.isPlatformConfigured()) {
            throw new IOException("No valid platform configuration available");
        }
    }

    private static String getVFSHomeDirectory() throws FileNotFoundException {
        return USER_HOME + SshThread.getCurrentThreadUser();
    }

    private static String getNFSHomeDirectory() throws FileNotFoundException {
        NativeAuthenticationProvider nap = NativeAuthenticationProvider
            .getInstance();

        try {
            return nap.getHomeDirectory(SshThread.getCurrentThreadUser());
        } catch (IOException e) {
            throw new FileNotFoundException(e.getMessage());
        }
    }

    public static String translateNFSPath(String nfspath)
        throws FileNotFoundException {
        nfspath = nfspath.replace('\\', '/');

        String nfshome = getNFSHomeDirectory().replace('\\', '/');
        String vfshome = getVFSHomeDirectory();

        // First check for the userhome
        if (nfspath.startsWith(nfshome)) {
            nfspath = translateCanonicalPath(nfspath, nfshome);
            return nfspath.replaceFirst(nfshome, vfshome);
        }

        // Now lets translate from the available mounts
        Iterator it = vfsmounts.entrySet().iterator();
        Map.Entry entry;
        String mount;
        String path;
        VFSMount m;

        while (it.hasNext()) {
            entry = (Map.Entry) it.next();
            mount = (String) entry.getKey();
            m = (VFSMount) entry.getValue();
            path = m.getPath();

            if (nfspath.startsWith(path)) {
                nfspath = translateCanonicalPath(nfspath, path);
                return nfspath.replaceFirst(path, mount);
            }
        }

        if (nfspath.startsWith(vfsroot.getPath())) {
            nfspath = translateCanonicalPath(nfspath, vfsroot.getPath());
            path = nfspath.replaceFirst(vfsroot.getPath(), "");

            return (path.startsWith("/") ? path : ("/" + path));
        } else {
            throw new FileNotFoundException("The file could not be found");
        }
    }

    private static VFSMount getMount(String vfspath)
        throws FileNotFoundException, IOException {
        String vfshome = getVFSHomeDirectory();
        VFSMount m;

        if (vfspath.startsWith("/")) {
            if (vfspath.startsWith(vfshome)) {
                m = new VFSMount(vfshome, getNFSHomeDirectory());
                m.setPermissions(new VFSPermission(
                        SshThread.getCurrentThreadUser(), "rwx"));

                return m;
            } else {
                Iterator it = vfsmounts.entrySet().iterator();
                Map.Entry entry;
                String mount;

                while (it.hasNext()) {
                    entry = (Map.Entry) it.next();
                    mount = (String) entry.getKey();

                    if (vfspath.startsWith(mount)) {
                        return (VFSMount) entry.getValue();
                    }
                }

                if (vfsroot != null) {
                    return vfsroot;
                } else {
                    throw new FileNotFoundException("The path was not found");
                }
            }
        } else {
            m = new VFSMount(vfshome, getNFSHomeDirectory());
            m.setPermissions(new VFSPermission(vfshome.substring(
                        vfshome.lastIndexOf("/")), "rwx"));

            return m;
        }
    }

    public static String translateVFSPath(String vfspath)
        throws FileNotFoundException {
        // Translate any backslashes for sanity
        vfspath = vfspath.replace('\\', '/').trim();

        String nfshome = getNFSHomeDirectory().replace('\\', '/');
        String vfshome = getVFSHomeDirectory();

        if (vfspath.startsWith("/")) {
            if (vfspath.startsWith(vfshome)) {
                // Return the canonicalized system dependent path
                if (vfspath.length() > vfshome.length()) {
                    return translateCanonicalPath(nfshome
                        + (nfshome.endsWith("/") ? "" : "/")
                        + vfspath.substring(vfshome.length()), nfshome);
                } else {
                    return translateCanonicalPath(nfshome, nfshome);
                }
            }

            // The path is absolute and does not refer to the absolute USER_HOME
            // so we will look up using the platform.xml VFS mounts
            Iterator it = vfsmounts.entrySet().iterator();
            Map.Entry entry;
            String mount;
            String path;
            VFSMount m;

            while (it.hasNext()) {
                entry = (Map.Entry) it.next();
                mount = (String) entry.getKey();
                m = (VFSMount) entry.getValue();
                path = m.getPath();

                if (vfspath.startsWith(mount)) {
                    // Lets translate the path, making sure we do not move outside
                    // vfs with ..
                    String str = path + vfspath.substring(mount.length());
                    // vfspath.replaceFirst(mount,
                    //        path)
                    return translateCanonicalPath(str, path);
                }
            }

            // If we reached here then the VFS path did not refer to an optional mount
            // or the users home directory, so lets attempt to use the VFS root is there
            // is one defined
            if (vfsroot != null) {
                path = vfsroot.getPath()
                    + (vfsroot.getPath().endsWith("/") ? vfspath.substring(1)
                                                       : vfspath);

                return translateCanonicalPath(path, vfsroot.getPath());
            } else {
                throw new FileNotFoundException("The file could not be found");
            }
        } else {
            String path = nfshome + (nfshome.endsWith("/") ? "" : "/")
                + vfspath;

            return translateCanonicalPath(path, nfshome);
        }
    }

    public static String translateCanonicalPath(String path, String securemount)
        throws FileNotFoundException {
        try {

            log.debug("Translating path " + path + " against secure mount " + securemount);

            File f = new File(path);
            String canonical = f.getCanonicalPath().replace('\\', '/');

            File f2 = new File(securemount);
            String canonical2 = f2.getCanonicalPath().replace('\\', '/');

            // Verify that the canonical path does not exit out of the mount
            if (canonical.startsWith(canonical2)) {
                return canonical;
            } else {
                throw new FileNotFoundException("The path could not be found");
            }
        } catch (IOException ex) {
            throw new FileNotFoundException("The path could not be found");
        }
    }

    public boolean makeDirectory(String path)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        verifyPermissions(SshThread.getCurrentThreadUser(), path, "rw");

        File f = new File(path);

        return f.mkdir();
    }

    public VFSPermission getVFSPermission(String path)
        throws FileNotFoundException, IOException {
        VFSMount mount = getMount(translateNFSPath(path));

        if (mount.getPermissions().containsKey(SshThread.getCurrentThreadUser())) {
            return (VFSPermission) mount.getPermissions().get(SshThread
                .getCurrentThreadUser());
        } else {
            return (VFSPermission) mount.getPermissions().get("default");
        }
    }

    public FileAttributes getFileAttributes(byte[] handle)
        throws IOException, InvalidHandleException {
        String shandle = new String(handle);

        if (openFiles.containsKey(shandle)) {
            Object obj = openFiles.get(shandle);
            File f;

            if (obj instanceof OpenFile) {
                f = ((OpenFile) obj).getFile();
            } else if (obj instanceof OpenDirectory) {
                f = ((OpenDirectory) obj).getFile();
            } else {
                throw new IOException("Unexpected open file handle");
            }

            VFSPermission permissions = getVFSPermission(f.getAbsolutePath());

            if (permissions == null) {
                throw new IOException("No default permissions set");
            }

            FileAttributes attrs = new FileAttributes();
            attrs.setSize(new UnsignedInteger64(String.valueOf(f.length())));
            attrs.setTimes(new UnsignedInteger32(f.lastModified() / 1000),
                new UnsignedInteger32(f.lastModified() / 1000));

            boolean canExec = true;

            try {
                if (System.getSecurityManager() != null) {
                    System.getSecurityManager().checkExec(f.getCanonicalPath());
                }
            } catch (SecurityException ex1) {
                canExec = false;
            }

            attrs.setPermissions((((f.canRead() && permissions.canRead()) ? "r"
                                                                          : "-")
                + ((f.canWrite() && permissions.canWrite()) ? "w" : "-")
                + ((canExec && permissions.canExecute()) ? "x" : "-")));
            attrs.setPermissions(new UnsignedInteger32(attrs.getPermissions()
                                                            .longValue()
                    | (f.isDirectory() ? FileAttributes.S_IFDIR
                                       : FileAttributes.S_IFREG)));

            return attrs;
        } else {
            throw new InvalidHandleException("The handle is invalid");
        }
    }

    public FileAttributes getFileAttributes(String path)
        throws IOException, FileNotFoundException {
        // Look up the VFS mount attributes
        File f = new File(path);

        VFSPermission permissions = getVFSPermission(path);

        if (permissions == null) {
            throw new IOException("No default permissions set");
        }

        FileAttributes attrs = new FileAttributes();
        attrs.setSize(new UnsignedInteger64(String.valueOf(f.length())));
        attrs.setTimes(new UnsignedInteger32(f.lastModified() / 1000),
            new UnsignedInteger32(f.lastModified() / 1000));

        boolean canExec = true;

        try {
            if (System.getSecurityManager() != null) {
                System.getSecurityManager().checkExec(f.getCanonicalPath());
            }
        } catch (SecurityException ex1) {
            canExec = false;
        }

        attrs.setPermissions((((f.canRead() && permissions.canRead()) ? "r" : "-")
            + ((f.canWrite() && permissions.canWrite()) ? "w" : "-")
            + ((canExec && permissions.canExecute()) ? "x" : "-")));

        attrs.setPermissions(new UnsignedInteger32(attrs.getPermissions()
                                                        .longValue()
                | (f.isDirectory() ? FileAttributes.S_IFDIR
                                   : FileAttributes.S_IFREG)));

        return attrs;
    }

    public byte[] openDirectory(String path)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        File f = new File(path);

        verifyPermissions(SshThread.getCurrentThreadUser(), path, "r");

        if (f.exists()) {
            if (f.isDirectory()) {
                openFiles.put(f.toString(), new OpenDirectory(f));

                return f.toString().getBytes();
            } else {
                throw new IOException(translateNFSPath(path)
                    + " is not a directory");
            }
        } else {
            throw new FileNotFoundException(translateNFSPath(path)
                + " does not exist");
        }
    }

    public File[] readDirectory(byte[] handle)
        throws InvalidHandleException, EOFException {
        String shandle = new String(handle);

        if (openFiles.containsKey(shandle)) {
            Object obj = openFiles.get(shandle);

            if (obj instanceof OpenDirectory) {
                OpenDirectory dir = (OpenDirectory) obj;
                int pos = dir.getPosition();
                File[] children = dir.getChildren();
                int count = ((children.length - pos) < 100)
                    ? (children.length - pos) : 100;

                if (count > 0) {
                    File[] files = new File[count];
                    System.arraycopy(children, pos, files, 0, files.length);
                    dir.setPosition(pos + files.length);

                    return files;
                } else {
                    throw new EOFException("There are no more files");
                }
            } else {
                throw new InvalidHandleException(
                    "Handle is not an open directory");
            }
        } else {
            throw new InvalidHandleException("The handle is invalid");
        }
    }

    public byte[] openFile(String path, UnsignedInteger32 flags,
        FileAttributes attrs)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        File f = new File(path);

        verifyPermissions(SshThread.getCurrentThreadUser(), path, "r");

        // Check if the file does not exist and process according to flags
        if (!f.exists()) {
            if ((flags.intValue() & NativeFileSystemProvider.OPEN_CREATE) == NativeFileSystemProvider.OPEN_CREATE) {
                // The file does not exist and the create flag is present so lets create it
                if (!f.createNewFile()) {
                    throw new IOException(translateNFSPath(path)
                        + " could not be created");
                }
            } else {
                // The file does not exist and no create flag present
                throw new FileNotFoundException(translateNFSPath(path)
                    + " does not exist");
            }
        } else {
            if (((flags.intValue() & NativeFileSystemProvider.OPEN_CREATE) == NativeFileSystemProvider.OPEN_CREATE)
                    && ((flags.intValue()
                    & NativeFileSystemProvider.OPEN_EXCLUSIVE) == NativeFileSystemProvider.OPEN_EXCLUSIVE)) {
                // The file exists but the EXCL flag is set which requires that the
                // file should not exist prior to creation, so throw a status exception
                throw new IOException(translateNFSPath(path)
                    + " already exists");
            }
        }

        // The file now exists so open the file according to the flags yb building the relevant
        // flags for the RandomAccessFile class
        String mode = "r"
            + (((flags.intValue() & NativeFileSystemProvider.OPEN_WRITE) == NativeFileSystemProvider.OPEN_WRITE)
            ? "ws" : "");

        RandomAccessFile raf = new RandomAccessFile(f, mode);

        // Determine whether we need to truncate the file
        if (((flags.intValue() & NativeFileSystemProvider.OPEN_CREATE) == NativeFileSystemProvider.OPEN_CREATE)
                && ((flags.intValue() & NativeFileSystemProvider.OPEN_TRUNCATE) == NativeFileSystemProvider.OPEN_TRUNCATE)) {
            // Set the length to zero
            raf.setLength(0);
        }

        // Record the open file
        openFiles.put(raf.toString(), new OpenFile(f, raf, flags));

        // Return the handle
        return raf.toString().getBytes();
    }

    public byte[] readFile(byte[] handle, UnsignedInteger64 offset,
        UnsignedInteger32 len)
        throws InvalidHandleException, EOFException, IOException {
        String shandle = new String(handle);

        if (openFiles.containsKey(shandle)) {
            Object obj = openFiles.get(shandle);

            if (obj instanceof OpenFile) {
                OpenFile file = (OpenFile) obj;

                if ((file.getFlags().intValue()
                        & NativeFileSystemProvider.OPEN_READ) == NativeFileSystemProvider.OPEN_READ) {
                    byte[] buf = new byte[len.intValue()];

                    if (file.getRandomAccessFile().getFilePointer() != offset
                            .longValue()) {
                        file.getRandomAccessFile().seek(offset.longValue());
                    }

                    int read = file.getRandomAccessFile().read(buf);

                    if (read >= 0) {
                        if (read == buf.length) {
                            return buf;
                        } else {
                            byte[] tmp = new byte[read];
                            System.arraycopy(buf, 0, tmp, 0, read);

                            return tmp;
                        }
                    } else {
                        throw new EOFException("The file is EOF");
                    }
                } else {
                    throw new InvalidHandleException(
                        "The file handle was not opened for reading");
                }
            } else {
                throw new InvalidHandleException("Handle is not an open file");
            }
        } else {
            throw new InvalidHandleException("The handle is invalid");
        }
    }

    public void writeFile(byte[] handle, UnsignedInteger64 offset, byte[] data,
        int off, int len) throws InvalidHandleException, IOException {
        String shandle = new String(handle);

        if (openFiles.containsKey(shandle)) {
            Object obj = openFiles.get(shandle);

            if (obj instanceof OpenFile) {
                OpenFile file = (OpenFile) obj;

                if ((file.getFlags().intValue()
                        & NativeFileSystemProvider.OPEN_WRITE) == NativeFileSystemProvider.OPEN_WRITE) {
                    if ((file.getFlags().intValue()
                            & NativeFileSystemProvider.OPEN_APPEND) == NativeFileSystemProvider.OPEN_APPEND) {
                        // Force the data to be written to the end of the file by seeking to the end
                        file.getRandomAccessFile().seek(file.getRandomAccessFile()
                                                            .length());
                    } else if (file.getRandomAccessFile().getFilePointer() != offset
                            .longValue()) {
                        // Move the file pointer if its not in the write place
                        file.getRandomAccessFile().seek(offset.longValue());
                    }

                    file.getRandomAccessFile().write(data, off, len);
                } else {
                    throw new InvalidHandleException(
                        "The file was not opened for writing");
                }
            } else {
                throw new InvalidHandleException("Handle is not an open file");
            }
        } else {
            throw new InvalidHandleException("The handle is invalid");
        }
    }

    public void closeFile(byte[] handle)
        throws InvalidHandleException, IOException {
        String shandle = new String(handle);

        if (openFiles.containsKey(shandle)) {
            Object obj = openFiles.get(shandle);

            if (obj instanceof OpenDirectory) {
                openFiles.remove(shandle);
            } else if (obj instanceof OpenFile) {
                ((OpenFile) obj).getRandomAccessFile().close();
                openFiles.remove(shandle);
            } else {
                throw new InvalidHandleException("Internal server error");
            }
        } else {
            throw new InvalidHandleException("The handle is invalid");
        }
    }

    public void removeFile(String path)
        throws PermissionDeniedException, IOException, FileNotFoundException {
        File f = new File(path);

        if (f.exists()) {
            try {
                if (f.isFile()) {
                    if (!f.delete()) {
                        throw new IOException("Failed to delete "
                            + translateNFSPath(path));
                    }
                } else {
                    throw new IOException(translateNFSPath(path)
                        + " is a directory, use remove directory command to remove");
                }
            } catch (SecurityException se) {
                throw new PermissionDeniedException("Permission denied");
            }
        } else {
            throw new FileNotFoundException(translateNFSPath(path)
                + " does not exist");
        }
    }

    public void renameFile(String oldpath, String newpath)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        File f = new File(oldpath);

        verifyPermissions(SshThread.getCurrentThreadUser(), oldpath, "rw");
        verifyPermissions(SshThread.getCurrentThreadUser(), newpath, "rw");

        if (f.exists()) {
            File f2 = new File(newpath);

            if (!f2.exists()) {
                if (!f.renameTo(f2)) {
                    throw new IOException("Failed to rename file "
                        + translateNFSPath(oldpath));
                }
            } else {
                throw new IOException(translateNFSPath(newpath)
                    + " already exists");
            }
        } else {
            throw new FileNotFoundException(translateNFSPath(oldpath)
                + " does not exist");
        }
    }

    public void removeDirectory(String path)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        File f = new File(path);

        verifyPermissions(SshThread.getCurrentThreadUser(), path, "rw");

        if (f.isDirectory()) {
            if (f.exists()) {
                if (f.listFiles().length == 0) {
                    if (!f.delete()) {
                        throw new IOException("Failed to remove directory "
                            + translateNFSPath(path));
                    }
                } else {
                    throw new IOException(translateNFSPath(path)
                        + " is not an empty directory");
                }
            } else {
                throw new FileNotFoundException(translateNFSPath(path)
                    + " does not exist");
            }
        } else {
            throw new IOException(translateNFSPath(path)
                + " is not a directory");
        }
    }

    public void setFileAttributes(String path, FileAttributes attrs)
        throws PermissionDeniedException, IOException, FileNotFoundException {
    }

    public void setFileAttributes(byte[] handle, FileAttributes attrs)
        throws PermissionDeniedException, IOException, InvalidHandleException {
    }

    public File readSymbolicLink(String path)
        throws UnsupportedFileOperationException, FileNotFoundException,
            IOException, PermissionDeniedException {
        throw new UnsupportedFileOperationException(
            "Symbolic links are not supported by the Virtual File System");
    }

    public void createSymbolicLink(String link, String target)
        throws UnsupportedFileOperationException, FileNotFoundException,
            IOException, PermissionDeniedException {
        throw new UnsupportedFileOperationException(
            "Symbolic links are not supported by the Virtual File System");
    }

    public void verifyPermissions(String username, String path,
        String permissions)
        throws PermissionDeniedException, FileNotFoundException, IOException {
        String vfspath = translateNFSPath(path);
        VFSMount mount = getMount(vfspath);

        VFSPermission perm;

        if (mount.getPermissions().containsKey(SshThread.getCurrentThreadUser())) {
            perm = (VFSPermission) mount.getPermissions().get(SshThread
                    .getCurrentThreadUser());
        } else if (mount.getPermissions().containsKey("default")) {
            perm = (VFSPermission) mount.getPermissions().get("default");
        } else {
            throw new PermissionDeniedException("No permissions set for mount");
        }

        if (!perm.verifyPermissions(permissions)) {
            throw new PermissionDeniedException("Permission denied for "
                + translateNFSPath(path));
        }
    }

    class OpenFile {
        File f;
        RandomAccessFile raf;
        UnsignedInteger32 flags;

        public OpenFile(File f, RandomAccessFile raf, UnsignedInteger32 flags) {
            this.f = f;
            this.raf = raf;
            this.flags = flags;
        }

        public File getFile() {
            return f;
        }

        public RandomAccessFile getRandomAccessFile() {
            return raf;
        }

        public UnsignedInteger32 getFlags() {
            return flags;
        }
    }

    class OpenDirectory {
        File f;
        File[] children;
        int readpos = 0;

        public OpenDirectory(File f) {
            this.f = f;
            this.children = f.listFiles();
        }

        public File getFile() {
            return f;
        }

        public File[] getChildren() {
            return children;
        }

        public int getPosition() {
            return readpos;
        }

        public void setPosition(int readpos) {
            this.readpos = readpos;
        }
    }
}
