/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.subsystem;

import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.transport.InvalidMessageException;

import com.sshtools.j2ssh.util.StartStopState;

import org.apache.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.sshtools.j2ssh.connection.ChannelState;

import com.sshtools.j2ssh.session.SessionChannelClient;

public abstract class SubsystemClient implements Runnable {
    private static Logger log = Logger.getLogger(SubsystemClient.class);
    private InputStream in;
    private OutputStream out;
    private Thread thread;
    private String name;
    private StartStopState state = new StartStopState(StartStopState.STOPPED);
    protected SubsystemMessageStore messageStore;
    protected SessionChannelClient session;

    public SubsystemClient(String name) {
        this.name = name;
        messageStore = new SubsystemMessageStore();
    }

    public SubsystemClient(String name, SubsystemMessageStore messageStore) {
        this.name = name;
        this.messageStore = messageStore;
    }

    public boolean isClosed() {
        return state.getValue() == StartStopState.STOPPED;
    }

    public void setSessionChannel(SessionChannelClient session) {
      this.session = session;
      this.in = session.getInputStream();
      this.out = session.getOutputStream();
    }

    public boolean start() throws IOException {
        thread = new SshThread(this, name + " subsystem", true);
        if(session==null)
          throw new IOException("No valid session is attached to the subsystem!");

        if(session.getState().getValue()!=ChannelState.CHANNEL_OPEN)
          throw new IOException("The session is not open!");

        thread.start();
        return onStart();
    }

    protected abstract boolean onStart() throws IOException;

    public String getName() {
        return name;
    }

    protected void sendMessage(SubsystemMessage msg)
        throws InvalidMessageException, IOException {
        log.info("Sending " + msg.getMessageName() + " subsystem message");

        byte[] msgdata = msg.toByteArray();

        // Write the message length
        out.write(ByteArrayWriter.encodeInt(msgdata.length));

        // Write the message data
        out.write(msgdata);
    }

    public void run() {
        int read;
        int len;
        int pos;
        byte[] buffer = new byte[4];
        byte[] msg;

        state.setValue(StartStopState.STARTED);


        try {
            // read the first four bytes of data to determine the susbsytem
            // message length
            while (state.getValue() == StartStopState.STARTED
                    && session.getState().getValue()==ChannelState.CHANNEL_OPEN) {
                read = in.read(buffer);

                if (read > 0) {
                    len = (int)ByteArrayReader.readInt(buffer, 0);
                    msg = new byte[len];
                    pos = 0;

                    while (pos < len)
                        pos += in.read(msg, pos, msg.length - pos);

                    messageStore.addMessage(msg);
                    msg = null;
                }
            }
        } catch (IOException ioe) {
            log.fatal("Subsystem message loop failed!", ioe);
        }

        thread = null;
    }

    public void stop() throws IOException {
        state.setValue(StartStopState.STOPPED);
        in.close();
        out.close();
        session.close();
    }
}
