/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;


public abstract class SshMessage {
    // The message Id of the message
    private int messageId;

    public SshMessage(int messageId) {
        // Save the message id
        this.messageId = messageId;
    }

    public final int getMessageId() {
        return messageId;
    }

    public abstract String getMessageName();

    public final byte[] toByteArray() throws InvalidMessageException {
        // Create a writer object to construct the array
        ByteArrayWriter baw = new ByteArrayWriter();

        // Write the message id
        baw.write(messageId);

        // Call the abstract method so subclasses classes can add their data
        constructByteArray(baw);

        // Return the array
        return baw.toByteArray();
    }

    protected final void fromByteArray(ByteArrayReader data)
        throws InvalidMessageException {
        // Skip the first 5 bytes as this contains the packet length and payload
        // length fields
        data.skip(5);

        int id = data.read();

        if (id != messageId) {
            throw new InvalidMessageException("The message id "
                + String.valueOf(id)
                + " is not the same as the message implementation id "
                + String.valueOf(messageId));
        }

        // Call abstract method for subclasses to extract the message specific data
        constructMessage(data);
    }

    public static Integer getMessageId(byte[] msgdata) {
        return new Integer(msgdata[5]);
    }

    protected abstract void constructByteArray(ByteArrayWriter baw)
        throws InvalidMessageException;

    protected abstract void constructMessage(ByteArrayReader bar)
        throws InvalidMessageException;
}
