/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport.publickey;

import com.sshtools.j2ssh.io.ByteArrayReader;

import com.sshtools.j2ssh.transport.AlgorithmNotSupportedException;

import org.apache.log4j.Logger;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import java.util.Iterator;


public class SshPublicKeyFile {
    private static Logger log = Logger.getLogger(SshPublicKeyFile.class);
    private SshPublicKeyFormat format;
    private byte[] keyblob;
    private String comment;

    protected SshPublicKeyFile(byte[] keyblob, SshPublicKeyFormat format) {
        this.keyblob = keyblob;
        this.format = format;
    }

    public byte[] getBytes() {
        return format.formatKey(keyblob);
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        this.comment = comment;
    }

    public byte[] getKeyBlob() {
        return keyblob;
    }

    public static SshPublicKeyFile create(SshPublicKey key,
        SshPublicKeyFormat format) {
        SshPublicKeyFile file = new SshPublicKeyFile(key.getEncoded(), format);
        file.setComment(format.getComment());

        return file;
    }

    public static SshPublicKeyFile parse(File keyfile)
        throws InvalidSshKeyException, IOException {
        FileInputStream in = new FileInputStream(keyfile);
        byte[] data = new byte[in.available()];
        in.read(data);
        in.close();

        return parse(data);
    }

    public static SshPublicKeyFile parse(byte[] formattedKey)
        throws InvalidSshKeyException {
        log.info("Parsing public key file");

        // Try the default private key format
        SshPublicKeyFormat format;

        format = SshPublicKeyFormatFactory.newInstance(SshPublicKeyFormatFactory
                .getDefaultFormatType());

        boolean valid = format.isFormatted(formattedKey);

        if (!valid) {
            log.info(
                "Public key is not in the default format, attempting parse with other supported formats");

            Iterator it = SshPublicKeyFormatFactory.getSupportedFormats()
                                                   .iterator();
            String ft;

            while (it.hasNext() && !valid) {
                ft = (String) it.next();
                log.debug("Attempting " + ft);
                format = SshPublicKeyFormatFactory.newInstance(ft);
                valid = format.isFormatted(formattedKey);
            }
        }

        if (valid) {
            SshPublicKeyFile file = new SshPublicKeyFile(format.getKeyBlob(
                        formattedKey), format);
            file.setComment(format.getComment());

            return file;
        } else {
            throw new InvalidSshKeyException(
                "The key format is not a supported format");
        }
    }

    public String getAlgorithm() {
        return ByteArrayReader.readString(keyblob, 0);
    }

    public void setFormat(SshPublicKeyFormat newFormat)
        throws InvalidSshKeyException {
        if (newFormat.supportsAlgorithm(getAlgorithm())) {
            newFormat.setComment(format.getComment());
            this.format = newFormat;
        } else {
            throw new InvalidSshKeyException(
                "The format does not support the public key algorithm");
        }
    }

    public SshPublicKeyFormat getFormat() {
        return format;
    }

    public SshPublicKey toPublicKey() throws IOException {
        ByteArrayReader bar = new ByteArrayReader(keyblob);
        String type = bar.readString();
        SshKeyPair pair = SshKeyPairFactory.newInstance(type);

        return pair.decodePublicKey(keyblob);
    }

    public String toString() {
        return new String(format.formatKey(keyblob));
    }
}
