/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport.publickey.dsa;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.transport.publickey.InvalidSshKeyException;
import com.sshtools.j2ssh.transport.publickey.InvalidSshKeySignatureException;
import com.sshtools.j2ssh.transport.publickey.SshPublicKey;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.math.BigInteger;

import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.DSAPublicKey;
import java.security.spec.DSAPublicKeySpec;


public class SshDssPublicKey extends SshPublicKey {
    private static Logger log = Logger.getLogger(SshDssPublicKey.class);
    private DSAPublicKey pubkey;

    public SshDssPublicKey(DSAPublicKey key) {
        this.pubkey = key;
    }

    public SshDssPublicKey(byte[] key) throws InvalidSshKeyException {
        try {
            DSAPublicKeySpec dsaKey;

            // Extract the key information
            ByteArrayReader bar = new ByteArrayReader(key);

            String header = bar.readString();

            if (!header.equals(getAlgorithmName())) {
                throw new InvalidSshKeyException();
            }

            BigInteger p = bar.readBigInteger();
            BigInteger q = bar.readBigInteger();
            BigInteger g = bar.readBigInteger();
            BigInteger y = bar.readBigInteger();

            dsaKey = new DSAPublicKeySpec(y, p, q, g);

            KeyFactory kf = KeyFactory.getInstance("DSA");
            pubkey = (DSAPublicKey) kf.generatePublic(dsaKey);
        } catch (Exception e) {
            throw new InvalidSshKeyException();
        }
    }

    public String getAlgorithmName() {
        return "ssh-dss";
    }

    public int getBitLength() {
        return pubkey.getY().bitLength();
    }

    public byte[] getEncoded() {
        try {
            ByteArrayWriter baw = new ByteArrayWriter();

            baw.writeString(getAlgorithmName());
            baw.writeBigInteger(pubkey.getParams().getP());
            baw.writeBigInteger(pubkey.getParams().getQ());
            baw.writeBigInteger(pubkey.getParams().getG());
            baw.writeBigInteger(pubkey.getY());

            return baw.toByteArray();
        } catch (IOException ioe) {
            return null;
        }
    }

    public boolean verifySignature(byte[] signature, byte[] data)
        throws InvalidSshKeySignatureException {
        try {
            log.debug("Verifying host key signature");
            log.debug("Signature length is " + String.valueOf(signature.length));

            // Check for differing version of the transport protocol
            if (signature.length != 40) {
                ByteArrayReader bar = new ByteArrayReader(signature);

                byte[] sig = bar.readBinaryString();

                //log.debug("Signature blob is " + new String(sig));
                String header = new String(sig);
                log.debug("Header is " + header);

                if (!header.equals("ssh-dss")) {
                    throw new InvalidSshKeySignatureException();
                }

                signature = bar.readBinaryString();
                log.debug("Read signature from blob: " + new String(signature));
            }

            String hex = "";

            for (int i = 0; i < signature.length; i++)
                hex += (Integer.toHexString(signature[i] & 0xFF) + ": ");

            log.debug("Decoding signature " + hex);
            hex = "";

            for (int i = 0; i < signature.length; i++)
                hex += (Integer.toString(signature[i]) + ": ");

            log.debug("Decoding signature " + hex);

            byte[] encoded;

            // Determine the encoded length of the big integers
            int rlen = (((signature[0] & 0x80) == 0x80) ? 0x15 : 0x14);
            log.debug("rlen=" + String.valueOf(rlen));

            int slen = (((signature[20] & 0x80) == 0x80) ? 0x15 : 0x14);
            log.debug("slen=" + String.valueOf(slen));

            byte[] asn1r = { 0x30, (byte) (rlen + slen + 4), 0x02, (byte) rlen };
            byte[] asn1s = { 0x02, (byte) slen };

            // Create the encoded byte array
            encoded = new byte[asn1r.length + rlen + asn1s.length + slen];

            // Copy the data and encode it into the array
            System.arraycopy(asn1r, 0, encoded, 0, asn1r.length);

            // Copy the integer inserting a zero byte if signed
            int roffset = (((signature[0] & 0x80) == 0x80) ? 1 : 0);
            System.arraycopy(signature, 0, encoded, asn1r.length + roffset, 20);
            System.arraycopy(asn1s, 0, encoded, asn1r.length + roffset + 20,
                asn1s.length);

            int soffset = (((signature[20] & 0x80) == 0x80) ? 1 : 0);
            System.arraycopy(signature, 20, encoded,
                asn1r.length + roffset + 20 + asn1s.length + soffset, 20);

            Signature s = Signature.getInstance("SHA1withDSA");
            s.initVerify(pubkey);
            s.update(data);

            return s.verify(encoded);
        } catch (NoSuchAlgorithmException nsae) {
            throw new InvalidSshKeySignatureException();
        } catch (InvalidKeyException ike) {
            throw new InvalidSshKeySignatureException();
        } catch (IOException ioe) {
            throw new InvalidSshKeySignatureException();
        } catch (SignatureException se) {
            throw new InvalidSshKeySignatureException();
        }
    }
}
