/*
 *  Sshtools - Application Framework
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.daemon.windows;

import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.platform.NativeAuthenticationProvider;
import com.sshtools.j2ssh.platform.NativeProcessProvider;

import org.apache.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.Map;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;


public class WindowsProcess extends NativeProcessProvider {
    static {
        System.loadLibrary("sshtools-daemon-win32");
    }

    private static Logger log = Logger.getLogger(WindowsProcess.class);
    WindowsAuthenticationInfo authInfo;
    WindowsProcessInfo processInfo;
    WindowsProcessInputStream input;
    WindowsProcessInputStream error;
    WindowsProcessOutputStream output;

    public WindowsProcess() {
    }

    public InputStream getInputStream() {
        return input;
    }

    public OutputStream getOutputStream() {
        return output;
    }

    public InputStream getStderrInputStream() {
        return error;
    }

    public void kill() {
        killProcess(processInfo);
    }

    public String getDefaultTerminalProvider() {
      return getNativeTerminalProvider();
    }

    public boolean supportsPseudoTerminal(String term) {
        return false;
    }

    public boolean allocatePseudoTerminal(String term, int cols, int rows,
        int width, int height, String modes) {
        return false;
    }

    public int waitForExitCode() {
        return waitForProcessExitCode(processInfo.getProcessHandle());
    }

    public boolean stillActive() {
        return isProcessActive(processInfo.getProcessHandle());
    }

    public void start() throws IOException {
        if (!resumeProcess(processInfo)) {
            throw new IOException("Process could not be resumed");
        }
    }


    public boolean createProcess(String command, Map environment)
        throws IOException {
        if (!(Thread.currentThread() instanceof SshThread)) {
            log.error(
                "Calling thread is not an SshThread, failed to read username");

            return false;
        }

        SshThread thread = (SshThread) Thread.currentThread();

        if (thread.containsProperty("Win32UserAuthInfo")) {
           authInfo = (WindowsAuthenticationInfo) thread.getProperty(
                            "Win32UserAuthInfo");

           WindowsAuthentication auth =
               (WindowsAuthentication) NativeAuthenticationProvider.getInstance();

        String env = "";
        String username = thread.getUsername();

        /**
         * Determine the users home directory, first get the home directory setting,
         * under windows this can be empty, so we will attempt to get the profile
         * path instead in that case, finally if no profile path can be found, the
         * "DefaultHomeDir" platform setting will be used
         * */
        String directory = auth.getHomeDirectory(username);

        if(directory==null || directory.trim().equals("")) {
          // Try to get the profile directory instead
          directory = auth.getNativeProfileDirectory(authInfo.getLogonToken());
        }
        if (directory == null  || directory.equals("")) {
          directory = ConfigurationLoader.getPlatformConfiguration().getSetting("DefaultHomeDir");
        }


        // Create the process
        processInfo = createProcess(command, env, directory,
                    authInfo.getLogonToken());
        } else {
            // Create the process
            //			processInfo = createProcess(command, env, directory, 0);
            return false;
        }

        if (processInfo == null) {
            return false;
        }

        input = new WindowsProcessInputStream(processInfo.getStdoutReadHandle());
        output = new WindowsProcessOutputStream(processInfo.getStdinWriteHandle());

        error = new WindowsProcessInputStream(processInfo.getStderrReadHandle());

        return true;
    }

    private native int waitForProcessExitCode(int processHandle);

    private native boolean isProcessActive(int processHandle);

    private native WindowsProcessInfo createProcess(String command,
        String environment, String directory, int userTokenHandle);

    private native void killProcess(WindowsProcessInfo processInfo);

    private native String getNativeTerminalProvider();

    private native boolean resumeProcess(WindowsProcessInfo processInfo);
}
