/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.apache.log4j.Logger;
import com.sshtools.j2ssh.authentication.AuthenticationProtocolServer;
import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.SshConnectionProperties;
import com.sshtools.j2ssh.connection.ConnectionProtocol;
import com.sshtools.j2ssh.transport.TransportProtocol;
import com.sshtools.j2ssh.transport.TransportProtocolEventHandler;
import com.sshtools.j2ssh.transport.TransportProtocolServer;
import com.sshtools.j2ssh.util.StartStopState;

public abstract class SshServer {
  private static Logger log = Logger.getLogger(SshServer.class);
  private ConnectionListener listener = null;
  private ServerSocket server = null;
  private boolean shutdown = false;

  public SshServer() throws SshException {
    String serverId = System.getProperty("sshtools.serverid");

    if (serverId != null) {
      TransportProtocolServer.SOFTWARE_VERSION_COMMENTS = serverId;
    }

    if (!ConfigurationLoader.isServerConfigured()) {
      throw new SshException("Server configuration not available!");
    }

    if (!ConfigurationLoader.isPlatformConfigured()) {
      throw new SshException("Platfrom configuration not available");
    }

    if (ConfigurationLoader.getServerConfiguration().getServerHostKeys()
        .size() <= 0) {
      throw new SshException(
          "Server cannot start because there are no server host keys available");
    }
  }

  public void startServer() throws IOException {
    log.info("Starting server");
    startServerSocket();
    startCommandSocket();
  }

  protected void processCommand(int command, Socket client) throws IOException {
    if (command == 0x3a) {
      stopServer();
    }
  }

  protected void startCommandSocket() throws IOException {
    try {
      ServerSocket socket = new ServerSocket(ConfigurationLoader.
                                             getServerConfiguration()
                                             .getCommandPort(),
                                             50, InetAddress.getLocalHost());
      Socket client;

      while ( (client = socket.accept()) != null) {
        log.info("Command request received");

        // Read and process the command
        processCommand(client.getInputStream().read(), client);
        client.close();

        if (shutdown) {
          break;
        }
      }

      socket.close();
    }
    catch (Exception e) {
      if (!shutdown) {
        log.fatal("The command socket failed", e);
      }
    }
  }

  protected void startServerSocket() throws IOException {
    listener = new ConnectionListener(ConfigurationLoader.
                                      getServerConfiguration()
                                      .getListenAddress(),
                                      ConfigurationLoader.
                                      getServerConfiguration().getPort());
    listener.start();
  }

  protected void stopServer() throws IOException {
    log.info("Shutting down");
    listener.stop();
    listener = null;
    System.exit(0);
  }

  protected abstract void configureServices(ConnectionProtocol connection) throws
      IOException;

  protected void refuseSession(Socket socket) throws IOException {
    TransportProtocolServer transport = new TransportProtocolServer(true);
    transport.startTransportProtocol(socket, new SshConnectionProperties());
  }

  protected TransportProtocolServer createSession(Socket socket) throws
      IOException {

    log.debug("Initializing connection");

    InetAddress address = ( (InetSocketAddress) socket
                           .getRemoteSocketAddress()).getAddress();

    log.debug("Remote Hostname: " + address.getHostName());
    log.debug("Remote IP: " + address.getHostAddress());

    TransportProtocolServer transport = new TransportProtocolServer();

    // Create the Authentication Protocol
    AuthenticationProtocolServer authentication = new
        AuthenticationProtocolServer();

    // Create the Connection Protocol
    ConnectionProtocol connection = new ConnectionProtocol();

    // Configure the connections services
    configureServices(connection);

    // Allow the Connection Protocol to be accepted by the Authentication Protocol
    authentication.acceptService(connection);

    // Allow the Authentication Protocol to be accepted by the Transport Protocol
    transport.acceptService(authentication);

    transport.startTransportProtocol(socket,
                                     new SshConnectionProperties());

    return transport;

  }

  class ConnectionListener
      implements Runnable {
    private List activeConnections = new Vector();
    private Logger log = Logger.getLogger(ConnectionListener.class);
    private ServerSocket server;
    private String listenAddress;
    private Thread thread;
    private int maxConnections;
    private int port;
    private StartStopState state = new StartStopState(StartStopState.STOPPED);

    public ConnectionListener(String listenAddress, int port) {
      this.port = port;
      this.listenAddress = listenAddress;
    }

    public List getActiveConnections() {
      return activeConnections;
    }

    public void run() {
      try {
        log.debug("Starting connection listener thread");

        state.setValue(StartStopState.STARTED);

        server = new ServerSocket(port);

        Socket socket;
        maxConnections = ConfigurationLoader.getServerConfiguration()
            .getMaxConnections();

        boolean refuse = false;

        TransportProtocolEventHandler eventHandler =
            new TransportProtocolEventHandler() {

          public void onSocketTimeout(TransportProtocol transport,
                                      boolean stillConnected) {
          }

          public void onDisconnect(TransportProtocol transport) {

            synchronized (activeConnections) {
              log.info(transport.getRemoteAddress().getHostName() +
                       " connection closed");
              activeConnections.remove(transport);
            }
          }
        };

        while ( ( (socket = server.accept()) != null)
               && (state.getValue() == StartStopState.STARTED)) {

          log.debug("New connection requested");

          if (maxConnections < activeConnections.size()) {
            refuseSession(socket);
          }
          else {
            TransportProtocolServer transport =
                createSession(socket);

            log.info("Monitoring active session from "
                     + socket.getInetAddress().getCanonicalHostName());

            synchronized (activeConnections) {
              activeConnections.add(transport);
            }

            transport.addEventHandler(eventHandler);

          }

        }

        stop();

        log.info("Exiting connection listener thread");

      }
      catch (IOException ioe) {
        if (state.getValue() != StartStopState.STOPPED) {
          log.debug("The listening socket failed", ioe);
        }
      }
      finally {
        thread = null;
      }
    }

    public void start() {
      thread = new SshThread(this, "Connection listener", true);
      thread.start();
    }

    public void stop() {
      try {
        state.setValue(StartStopState.STOPPED);

        server.close();

        // Close all the connected sessions
        Iterator it = activeConnections.iterator();

        while (it.hasNext()) {
          TransportProtocol transport = (TransportProtocol) it.next();
          transport.disconnect("The server is shutting down");
        }
      }
      catch (IOException ioe) {
        log.warn("The listening socket failed to stop cleanly", ioe);
      }
    }

  }
}
