/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.configuration;

import com.sshtools.j2ssh.configuration.automate.AutomationConfiguration;

import com.sshtools.j2ssh.util.DynamicClassLoader;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.net.MalformedURLException;
import java.net.URL;

import java.security.AccessControlException;
import java.security.AccessController;
import java.security.SecureRandom;

import java.util.List;
import java.util.Properties;
import java.util.PropertyPermission;
import java.util.Vector;


public class ConfigurationLoader {
    private static Logger log = Logger.getLogger(ConfigurationLoader.class);
    private static SshAPIConfiguration config;
    private static ServerConfiguration server;
    private static PlatformConfiguration platform;
    private static AutomationConfiguration automation;
    private static String configDirectory;
    private static String configResource;
    private static String hostsResource;
    private static String serverResource;
    private static String platformResource;
    private static String automationResource;
    private static String homeDir;
    private static ClassLoader ext;
    private static ClassLoader clsLoader = null;
    private static SecureRandom rnd;
    private static String logfile = null;
    private static boolean initialized = false;
    private static boolean performingInitialization = false;
    private static Object initializationLock = new Object();

    static {
        // Get the sshtools.home system property. If system properties can not be
        // read then we are running in a sandbox
        try {
            homeDir = checkAndGetProperty("sshtools.home", null);
            configResource = checkAndGetProperty("sshtools.config", null);
            serverResource = checkAndGetProperty("sshtools.server", null);
            platformResource = checkAndGetProperty("sshtools.platform", null);
            automationResource = checkAndGetProperty("sshtools.automation", null);
            logfile = checkAndGetProperty("sshtools.logfile", null);

            try {
                if (logfile != null) {
                    setLogfile(logfile);
                }
            } catch (IOException ex) {
            }

            if (homeDir == null) {
                // Not sshtools.home so lets try the java.home instead
                homeDir = System.getProperty("java.home");

                if (!homeDir.endsWith(File.separator)) {
                    homeDir += File.separator;
                }

                configDirectory = homeDir + "lib";
            } else {

                File f = new File(homeDir);

                try {
                  homeDir = f.getCanonicalPath();
                }
                catch (IOException ex) {
                  log.warn("Failed to canonicalize home directory " + homeDir);
                }

                if (!homeDir.endsWith(File.separator)) {
                    homeDir += File.separator;
                }

                configDirectory = homeDir + "conf";
            }

            if (!configDirectory.endsWith(File.separator)) {
                configDirectory += File.separator;
            }

            if (configResource == null) {
                configResource = configDirectory + "sshtools.xml";
            } else {
                // Determine if the config file is in config dir or absolute
                File f = new File(configResource);

                if (!f.exists()) {
                    f = new File(configDirectory + configResource);

                    if (f.exists()) {
                        configResource = configDirectory + configResource;
                    } else {
                        configResource = configDirectory + "sshtools.xml";
                    }
                }
            }

            if (serverResource == null) {
                serverResource = configDirectory + "server.xml";
            } else {
                // Determine if the config file is in config dir or absolute
                File f = new File(serverResource);

                if (!f.exists()) {
                    f = new File(configDirectory + serverResource);

                    if (f.exists()) {
                        serverResource = configDirectory + serverResource;
                    } else {
                        serverResource = configDirectory + "server.xml";
                    }
                }
            }

            if (platformResource == null) {
                platformResource = configDirectory + "platform.xml";
            } else {
                // Determine if the config file is in config dir or absolute
                File f = new File(platformResource);

                if (!f.exists()) {
                    f = new File(configDirectory + platformResource);

                    if (f.exists()) {
                        platformResource = configDirectory + platformResource;
                    } else {
                        platformResource = configDirectory + "platform.xml";
                    }
                }
            }

            //loadPlatformConfiguration();
            if (automationResource == null) {
                automationResource = configDirectory + "automation.xml";
            } else {
                // Determine if the config file is in config dir or absolute
                File f = new File(automationResource);

                if (!f.exists()) {
                    f = new File(configDirectory + automationResource);

                    if (f.exists()) {
                        automationResource = configDirectory
                            + automationResource;
                    } else {
                        automationResource = configDirectory + "automation.xml";
                    }
                }
            }

            //loadAutomationConfiguration();
        } catch (AccessControlException ace) {
            log.info("No access to system properties. Must use "
                + "setAPIConfigurationResource() and / or "
                + "setServerConfigurationResource() before "
                + "using getAPIConfiguration() or getServerConfiguration()");
        }
    }

    protected ConfigurationLoader() {
    }

    public static SecureRandom getRND() {
        if (rnd == null) {
            rnd = new SecureRandom();
            rnd.nextInt();
        }

        return rnd;
    }

    public static String getVersionString(String projectname, String versionFile) {
        Properties properties = new Properties();
        String version = projectname;

        try {
            properties.load(loadFile(versionFile));

            String project = projectname.toLowerCase();
            String major = properties.getProperty(project + ".version.major");
            String minor = properties.getProperty(project + ".version.minor");
            String build = properties.getProperty(project + ".version.build");
            String type = properties.getProperty(project + ".project.type");

            if ((major != null) && (minor != null) && (build != null)) {
                version += (" " + major + "." + minor + "." + build);
            }

            if (type != null) {
                version += (" " + type);
            }
        } catch (Exception e) {
        }

        return version;
    }

    public static String checkAndGetProperty(String property,
        String defaultValue) {
        //  Check for access to sshtools.platform
        try {
            if (System.getSecurityManager() != null) {
                AccessController.checkPermission(new PropertyPermission(
                        property, "read"));
            }

            return System.getProperty(property, defaultValue);
        } catch (AccessControlException ace) {
            return defaultValue;
        }
    }

  public static void setLogfile(String logfile) throws IOException {
        ConfigurationLoader.logfile = logfile;

        if (logfile.equals("@console@")) {
            BasicConfigurator.configure();
        } else {
            RollingFileAppender log = new RollingFileAppender(new PatternLayout(
                        "%-5p [%t]: %m%n"), logfile, true);
            log.setMaxFileSize("100KB");
            BasicConfigurator.configure(log);
        }
    }

    public static void initialize(boolean force) {
        synchronized (initializationLock) {
            if (initialized && !force) {
                return;
            }

            performingInitialization = true;

            try {
              //  Look at the SSHTOOLS_HOME/lib/ext directory for additional
              //  Jar files to add to the classpath
              File dir = new File(homeDir + "lib" + File.separator + "ext");

              // Filter for .jar files
              FilenameFilter filter = new FilenameFilter() {
                public boolean accept(File dir, String name) {
                  return name.endsWith(".jar");
                }
              };

              // Get the list
              File[] children = dir.listFiles(filter);
              List classpath = new Vector();

              if (children != null) {
                for (int i = 0; i < children.length; i++) {
                  // Get filename of file or directory
                  log.info("Extension " + children[i].getAbsolutePath()
                           + " being added to classpath");
                  classpath.add(children[i].getAbsolutePath());
                }
              }

              // We need to setup the dynamic class loading with the extension jars
              ext = new DynamicClassLoader(ConfigurationLoader.class
                                           .getClassLoader(), classpath);
            }
            catch (AccessControlException ex) {
              log.info("Cannot access lib/ext directory, extension classes will not be loaded");
            }

            loadAPIConfiguration();
            loadServerConfiguration();
            loadPlatformConfiguration();
            loadAutomationConfiguration();

            // Now that we have the extension classes, initialize the factories
            com.sshtools.j2ssh.transport.cipher.SshCipherFactory.initialize();
            com.sshtools.j2ssh.authentication.SshAuthenticationClientFactory
            .initialize();
            com.sshtools.j2ssh.authentication.SshAuthenticationServerFactory
            .initialize();
            com.sshtools.j2ssh.transport.hmac.SshHmacFactory.initialize();
            com.sshtools.j2ssh.transport.compression.SshCompressionFactory
            .initialize();
            com.sshtools.j2ssh.transport.kex.SshKeyExchangeFactory.initialize();
            com.sshtools.j2ssh.transport.publickey.SshKeyPairFactory.initialize();

            performingInitialization = false;
            initialized = true;
        }
    }

    public static boolean isAPIConfigured() {
        return config != null;
    }

    public static SshAPIConfiguration getAPIConfiguration()
        throws ConfigurationException {
        synchronized (initializationLock) {
            if (!initialized && !performingInitialization) {
                initialize(true);
            }

            if (config == null) {
                throw new ConfigurationException(
                    "No valid API configuration is available");
            }

            return config;
        }
    }

    public static void setAPIConfigurationResource(String configResource) {
        ConfigurationLoader.configResource = configResource;
    }

    public static String getConfigurationDirectory() {
        return configDirectory;
    }

    public static Class getExtensionClass(String name)
        throws ClassNotFoundException {
        if (ext == null) {
            throw new ClassNotFoundException("Configuration not initialized");
        }

        return ext.loadClass(name);
    }

    public static String getHomeDirectory() {
        return homeDir;
    }

    public static boolean isPlatformConfigured() {
        return platform != null;
    }

    public static PlatformConfiguration getPlatformConfiguration()
        throws ConfigurationException {
        synchronized (initializationLock) {
            if (!initialized && !performingInitialization) {
                initialize(true);
            }

            if (platform == null) {
                throw new ConfigurationException(
                    "No valid platform configuration exists");
            }

            return platform;
        }
    }

    public static boolean isAutomationConfigured() {
        return automation != null;
    }

    public static AutomationConfiguration getAutomationConfiguration()
        throws ConfigurationException {
        synchronized (initializationLock) {
            if (!initialized && !performingInitialization) {
                initialize(true);
            }

            if (automation == null) {
                throw new ConfigurationException(
                    "Automation configuration not available");
            }

            return automation;
        }
    }

    public static void setContextClassLoader(ClassLoader clsLoader) {
        ConfigurationLoader.clsLoader = clsLoader;
    }

    public static ClassLoader getContextClassLoader() {
        return ConfigurationLoader.clsLoader;
    }

    public static boolean isContextClassLoader() {
        return (clsLoader != null);
    }

    public static boolean isServerConfigured() {
        return server != null;
    }

    public static ServerConfiguration getServerConfiguration()
        throws ConfigurationException {
        synchronized (initializationLock) {
            if (!initialized && !performingInitialization) {
                initialize(true);
            }

            if (server == null) {
                throw new ConfigurationException(
                    "No valid server configuration is available");
            }

            return server;
        }
    }

    public static void setServerConfigurationResource(String serverResource) {
        ConfigurationLoader.serverResource = serverResource;
    }

    public static void setPlatformConfigurationResource(String platformResource) {
        ConfigurationLoader.platformResource = platformResource;
    }

    public static void setHomeDirectory(String homeDir) {
        ConfigurationLoader.homeDir = homeDir;
    }

    public static void setAutomationConfigurationResource(
        String automationResource) {
        ConfigurationLoader.automationResource = automationResource;
    }

    public static InputStream loadFile(String filename)
        throws FileNotFoundException {
        FileInputStream in;
        log.info("Attempting to load " + filename);

        try {
            in = new FileInputStream(configDirectory + filename);

            return in;
        } catch (FileNotFoundException fnfe) {
            log.info(
                "Failed to load file from configuration directory, trying SSHTools home");
        }

        try {
            in = new FileInputStream(homeDir + filename);

            return in;
        } catch (FileNotFoundException fnfe) {
            log.info(
                "Failed to load file from SSHTools home directory, trying as absolute path");
        }

        in = new FileInputStream(filename);

        return in;
    }

    public static OutputStream saveFile(String filename)
        throws FileNotFoundException {
        // Look for the file in the config directory
        File f = new File(configDirectory + filename);

        if (f.exists()) {
            // Yes its there so create an outputstream to it
            return new FileOutputStream(f);
        } else {
            // Look for it absolute
            f = new File(filename);

            if (f.exists()) {
                return new FileOutputStream(filename); // yes so do absolute
            } else {
                // Determine whether the filename is absolute or not with a primitive check
                return new FileOutputStream((filename.indexOf(
                        File.pathSeparator) >= 0) ? filename
                                                  : (configDirectory + filename));
            }
        }
    }

    private static URL getResourceURL(String resource) {
        try {
            return new URL(resource);
        } catch (MalformedURLException murle) {
            try {
                return new File(resource).toURL();
            } catch (MalformedURLException murle2) {
                return null;
            }
        }
    }

    private static void loadAPIConfiguration() {
        // Load the configuration resource
        InputStream in = null;

        try {
            URL resource = getResourceURL(configResource);
            log.info("Loading api configuration from "
                + resource.toExternalForm());
            in = resource.openStream();
            config = new SshAPIConfiguration(in);
        } catch (Exception e) {
            log.warn("Api configuration not available");
            config = null;
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException ioe) {
                }
            }
        }
    }

    private static void loadPlatformConfiguration() {
        // Load the configuration resource
        InputStream in = null;

        try {
            URL resource = getResourceURL(platformResource);
            log.info("Loading platform configuration from "
                + resource.toExternalForm());
            in = resource.openStream();
            platform = new PlatformConfiguration(in);
        } catch (Exception e) {
            log.warn("Platform configuration not available");
            platform = null;
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException ioe) {
                }
            }
        }
    }

    private static void loadAutomationConfiguration() {
        // Load the configuration resource
        InputStream in = null;

        try {
            URL resource = getResourceURL(automationResource);
            log.info("Loading automation configuration from "
                + resource.toExternalForm());
            in = resource.openStream();
            automation = new AutomationConfiguration(in);
        } catch (Exception e) {
            log.warn("Automation configuration not available");
            automation = null;
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException ioe) {
                }
            }
        }
    }

    private static void loadServerConfiguration() {
        // Load the server resource
        InputStream in = null;

        try {
            URL resource = getResourceURL(serverResource);
            log.info("Loading server configuration from "
                + resource.toExternalForm());
            in = resource.openStream();
            server = new ServerConfiguration(in);
        } catch (Exception e) {
            log.warn("Server configuration not available");
            server = null;
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException ioe) {
                }
            }
        }
    }
}
