/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.configuration.automate;

import com.sshtools.j2ssh.transport.publickey.InvalidSshKeyException;

import java.io.IOException;

import java.util.Iterator;
import java.util.List;
import java.util.Vector;


public class RemoteIdentification {
    private String defaultName;
    private Vector rules = new Vector();
    private Class authorizedKeysFormat;
    private String defaultPath;

    public RemoteIdentification(String defaultName) {
        this.defaultName = defaultName;
    }

    protected List getRules() {
        return rules;
    }

    public String getDefaultName() {
        return defaultName;
    }

    public String getName(String ident) throws RemoteIdentificationException {
        boolean pass = false;
        Iterator it = rules.iterator();
        Vector passed = new Vector();
        RemoteIdentificationRule rule;
        String rulename = null;

        // Check all the rules
        while (it.hasNext()) {
            rule = (RemoteIdentificationRule) it.next();

            if (rule.testRule(ident)) {
                passed.add(rule);
            }
        }

        if (passed.size() > 0) {
            // Select the highest priority rule where 0=highest 10=lowest
            it = passed.iterator();

            RemoteIdentificationRule ret = null;

            while (it.hasNext()) {
                rule = (RemoteIdentificationRule) it.next();

                if (ret == null) {
                    ret = rule;
                } else {
                    if (rule.getPriority() < ret.getPriority()) {
                        ret = rule;
                    }
                }
            }

            if (ret.getName() != null) {
                return ret.getName();
            } else {
                return defaultName;
            }
        } else {
            throw new RemoteIdentificationException(
                "No rules exist to identify the remote host with ident string "
                + ident);
        }
    }

    protected void addRule(RemoteIdentificationRule rule) {
        rules.add(rule);
    }

    protected boolean testRules(String ident) {
        boolean pass = false;
        Iterator it = rules.iterator();
        RemoteIdentificationRule rule;

        while (it.hasNext() && !pass) {
            rule = (RemoteIdentificationRule) it.next();
            pass = rule.testRule(ident);
        }

        return pass;
    }

    protected void setAuthorizedKeysFormat(Class implementationClass) {
        authorizedKeysFormat = implementationClass;
    }

    protected void setAuthorizedKeysDefaultPath(String defaultPath) {
        this.defaultPath = defaultPath;
    }

    public String getAuthorizedKeysDefaultPath() {
        return defaultPath;
    }

    public AuthorizedKeysFormat getAuthorizedKeysFormat()
        throws RemoteIdentificationException {
        try {
            if (authorizedKeysFormat != null) {
                return (AuthorizedKeysFormat) authorizedKeysFormat.newInstance();
            } else {
                throw new RemoteIdentificationException(
                    "There is no authorized keys format set for this remote id");
            }
        } catch (Exception ex) {
            throw new RemoteIdentificationException("Failed to instansiate "
                + authorizedKeysFormat.getName());
        }
    }
}
