/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.forwarding;

import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelFactory;
import com.sshtools.j2ssh.connection.ConnectionProtocol;
import com.sshtools.j2ssh.connection.InvalidChannelException;

import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;

import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.TransportProtocolException;

import com.sshtools.j2ssh.util.InvalidStateException;
import com.sshtools.j2ssh.util.StartStopState;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.net.Socket;
import java.net.SocketPermission;
import java.net.InetSocketAddress;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

public class ForwardingClient
    implements ChannelFactory {
    private static Logger log = Logger.getLogger(ForwardingClient.class);
    private ConnectionProtocol connection;
    private List channelTypes = new Vector();
    private Map localForwardings = new HashMap();
    private Map remoteForwardings = new HashMap();
    private XDisplay xDisplay;
    private ForwardingConfiguration x11ForwardingConfiguration;

    public ForwardingClient(ConnectionProtocol connection) throws IOException {
        this.connection = connection;
        //channelTypes.add(ForwardingChannel.REMOTE_FORWARDING_CHANNEL);
        connection.addChannelFactory(ForwardingChannel.REMOTE_FORWARDING_CHANNEL, this);
        connection.addChannelFactory(ForwardingChannel.X11_FORWARDING_CHANNEL, this);
    }

    public List getChannelType() {
        return channelTypes;
    }

    public void enableX11Forwarding(XDisplay localDisplay) {
        xDisplay = localDisplay;
        x11ForwardingConfiguration = new ForwardingConfiguration("x11",
            "[Specified by server]", -1, xDisplay.getHost(), xDisplay.getPort());
    }

    public ForwardingConfiguration getX11ForwardingConfiguration() {
        return x11ForwardingConfiguration;
    }

    public boolean hasActiveConfigurations() {
        // First check the size
        if ( (localForwardings.size() == 0) && (remoteForwardings.size() == 0)) {
            return false;
        }

        Iterator it = localForwardings.values().iterator();

        while (it.hasNext()) {
            if ( ( (ForwardingConfiguration) it.next()).getState().getValue() ==
                StartStopState.STARTED) {
                return true;
            }
        }

        it = remoteForwardings.values().iterator();

        while (it.hasNext()) {
            if ( ( (ForwardingConfiguration) it.next()).getState().getValue() ==
                StartStopState.STARTED) {
                return true;
            }
        }

        return false;
    }

    public boolean hasActiveForwardings() {
        // First check the size
        if ( (localForwardings.size() == 0) && (remoteForwardings.size() == 0)) {
            return false;
        }

        Iterator it = localForwardings.values().iterator();

        while (it.hasNext()) {
            if ( ( (ForwardingConfiguration) it.next()).
                getActiveForwardingChannels()
                .size() > 0) {
                return true;
            }
        }

        it = remoteForwardings.values().iterator();

        while (it.hasNext()) {
            if ( ( (ForwardingConfiguration) it.next()).
                getActiveForwardingChannels()
                .size() > 0) {
                return true;
            }
        }

        return false;
    }

    public ForwardingConfiguration getLocalForwardingByAddress(
        String addressToBind, int portToBind) throws
        ForwardingConfigurationException {
        Iterator it = localForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                && (config.getPortToBind() == portToBind)) {
                return config;
            }
        }

        throw new ForwardingConfigurationException(
            "The configuration does not exist");
    }

    public ForwardingConfiguration getLocalForwardingByName(String name) throws
        ForwardingConfigurationException {
        if (!localForwardings.containsKey(name)) {
            throw new ForwardingConfigurationException(
                "The configuraiton does not exist!");
        }

        return (ForwardingConfiguration) localForwardings.get(name);
    }

    public Map getLocalForwardings() {
        return localForwardings;
    }

    public Map getRemoteForwardings() {
        return remoteForwardings;
    }

    public ForwardingConfiguration getRemoteForwardingByAddress(
        String addressToBind, int portToBind) throws
        ForwardingConfigurationException {
        Iterator it = remoteForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                && (config.getPortToBind() == portToBind)) {
                return config;
            }
        }

        throw new ForwardingConfigurationException(
            "The configuration does not exist");
    }

    public void removeLocalForwarding(String name) throws
        ForwardingConfigurationException {
        if (!localForwardings.containsKey(name)) {
            throw new ForwardingConfigurationException(
                "The name is not a valid forwarding configuration");
        }

        ForwardingListener listener = (ForwardingListener) localForwardings.get(
            name);

        if (listener.isRunning()) {
            stopLocalForwarding(name);
        }

        localForwardings.remove(name);
    }

    public void removeRemoteForwarding(String name) throws
        TransportProtocolException, ServiceOperationException,
        ForwardingConfigurationException {
        if (!remoteForwardings.containsKey(name)) {
            throw new ForwardingConfigurationException(
                "The name is not a valid forwarding configuration");
        }

        ForwardingListener listener = (ForwardingListener) remoteForwardings
            .get(name);

        if (listener.isRunning()) {
            stopRemoteForwarding(name);
        }

        remoteForwardings.remove(name);
    }

    public void addLocalForwarding(String uniqueName, String addressToBind,
                                   int portToBind, String hostToConnect,
                                   int portToConnect) throws
        ForwardingConfigurationException {
        // Check that the name does not exist
        if (localForwardings.containsKey(uniqueName)) {
            throw new ForwardingConfigurationException(
                "The configuration name already exists!");
        }

        // Check that the address to bind and port are not already being used
        Iterator it = localForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                && (config.getPortToBind() == portToBind)) {
                throw new ForwardingConfigurationException(
                    "The address and port are already in use");
            }
        }

        // Check the security mananger
        SecurityManager manager = System.getSecurityManager();

        if (manager != null) {
            try {
                manager.checkPermission(new SocketPermission(addressToBind
                    + ":" + String.valueOf(portToBind), "accept,listen"));
            }
            catch (SecurityException e) {
                throw new ForwardingConfigurationException(
                    "The security manager has denied listen permision on "
                    + addressToBind + ":" + String.valueOf(portToBind));
            }
        }

        // Create the configuration object
        localForwardings.put(uniqueName,
                             new ClientForwardingListener(uniqueName,
            connection, addressToBind,
            portToBind, hostToConnect, portToConnect));
    }

    public void addLocalForwarding(ForwardingConfiguration fwd) throws
        ForwardingConfigurationException {
        // Check that the name does not exist
        if (localForwardings.containsKey(fwd.getName())) {
            throw new ForwardingConfigurationException(
                "The configuration name already exists!");
        }

        // Check that the address to bind and port are not already being used
        Iterator it = localForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(fwd.getAddressToBind())
                && (config.getPortToBind() == fwd.getPortToBind())) {
                throw new ForwardingConfigurationException(
                    "The address and port are already in use");
            }
        }

        // Check the security mananger
        SecurityManager manager = System.getSecurityManager();

        if (manager != null) {
            try {
                manager.checkPermission(new SocketPermission(fwd
                    .getAddressToBind() + ":"
                    + String.valueOf(fwd.getPortToBind()), "accept,listen"));
            }
            catch (SecurityException e) {
                throw new ForwardingConfigurationException(
                    "The security manager has denied listen permision on "
                    + fwd.getAddressToBind() + ":"
                    + String.valueOf(fwd.getPortToBind()));
            }
        }

        // Create the configuration object
        localForwardings.put(fwd.getName(),
                             new ClientForwardingListener(fwd.getName(),
            connection,
            fwd.getAddressToBind(), fwd.getPortToBind(),
            fwd.getHostToConnect(), fwd.getPortToConnect()));
    }

    public void addRemoteForwarding(String uniqueName, String addressToBind,
                                    int portToBind, String hostToConnect,
                                    int portToConnect) throws
        ForwardingConfigurationException {
        // Check that the name does not exist
        if (remoteForwardings.containsKey(uniqueName)) {
            throw new ForwardingConfigurationException(
                "The remote forwaring configuration name already exists!");
        }

        // Check that the address to bind and port are not already being used
        Iterator it = remoteForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                && (config.getPortToBind() == portToBind)) {
                throw new ForwardingConfigurationException(
                    "The remote forwarding address and port are already in use");
            }
        }

        // Check the security mananger
        SecurityManager manager = System.getSecurityManager();

        if (manager != null) {
            try {
                manager.checkPermission(new SocketPermission(hostToConnect
                    + ":" + String.valueOf(portToConnect), "connect"));
            }
            catch (SecurityException e) {
                throw new ForwardingConfigurationException(
                    "The security manager has denied connect permision on "
                    + hostToConnect + ":" + String.valueOf(portToConnect));
            }
        }

        // Create the configuration object
        remoteForwardings.put(uniqueName,
                              new ForwardingConfiguration(uniqueName,
            addressToBind, portToBind,
            hostToConnect, portToConnect));
    }

    public void addRemoteForwarding(ForwardingConfiguration fwd) throws
        ForwardingConfigurationException {
        // Check that the name does not exist
        if (remoteForwardings.containsKey(fwd.getName())) {
            throw new ForwardingConfigurationException(
                "The remote forwaring configuration name already exists!");
        }

        // Check that the address to bind and port are not already being used
        Iterator it = remoteForwardings.values().iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(fwd.getAddressToBind())
                && (config.getPortToBind() == fwd.getPortToBind())) {
                throw new ForwardingConfigurationException(
                    "The remote forwarding address and port are already in use");
            }
        }

        // Check the security mananger
        SecurityManager manager = System.getSecurityManager();

        if (manager != null) {
            try {
                manager.checkPermission(new SocketPermission(fwd
                    .getHostToConnect() + ":"
                    + String.valueOf(fwd.getPortToConnect()), "connect"));
            }
            catch (SecurityException e) {
                throw new ForwardingConfigurationException(
                    "The security manager has denied connect permision on "
                    + fwd.getHostToConnect() + ":"
                    + String.valueOf(fwd.getPortToConnect()));
            }
        }

        // Create the configuration object
        remoteForwardings.put(fwd.getName(), fwd);
    }

    public Channel createChannel(String channelType, byte[] requestData) throws
        InvalidChannelException {

        if (channelType.equals(ForwardingChannel.X11_FORWARDING_CHANNEL)) {
            if (xDisplay == null) {
                throw new InvalidChannelException(
                    "Local display has not been set for X11 forwarding.");
            }

            try {
                ByteArrayReader bar = new ByteArrayReader(requestData);
                String originatingHost = bar.readString();
                int originatingPort = (int) bar.readInt();
                log.debug("Creating socket to " + x11ForwardingConfiguration.
                          getHostToConnect() + "/" + x11ForwardingConfiguration.
                                           getPortToConnect());
                Socket socket = new Socket(
                    x11ForwardingConfiguration.getHostToConnect(),
                    x11ForwardingConfiguration.getPortToConnect());

                // Create the channel adding it to the active channels
                ForwardingChannel channel = x11ForwardingConfiguration.
                    createForwardingChannel(
                    channelType,
                    x11ForwardingConfiguration.getHostToConnect(),
                    x11ForwardingConfiguration.getPortToConnect(),
                    originatingHost, originatingPort);

                channel.bindSocket(socket);

                return channel;
            }
            catch (IOException ioe) {
                throw new InvalidChannelException(ioe.getMessage());
            }
        }

        if (channelType.equals(ForwardingChannel.REMOTE_FORWARDING_CHANNEL)) {
            try {
                ByteArrayReader bar = new ByteArrayReader(requestData);
                String addressBound = bar.readString();
                int portBound = (int) bar.readInt();
                String originatingHost = bar.readString();
                int originatingPort = (int) bar.readInt();

                ForwardingConfiguration config = getRemoteForwardingByAddress(
                    addressBound,
                    portBound);

                Socket socket = new Socket(config.getHostToConnect(),
                                           config.getPortToConnect());

                // Create the channel adding it to the active channels
                ForwardingChannel channel = config.createForwardingChannel(
                    channelType,
                    config.getHostToConnect(), config.getPortToConnect(),
                    originatingHost, originatingPort);

                channel.bindSocket(socket);

                return channel;

            }
            catch (ForwardingConfigurationException fce) {
                throw new InvalidChannelException(
                    "No valid forwarding configuration was available for the request address");
            }
            catch (IOException ioe) {
                throw new InvalidChannelException(ioe.getMessage());
            }
        }

        throw new InvalidChannelException(
            "The server can only request a remote forwarding channel or an" +
            "X11 forwarding channel");

    }

    public void startLocalForwarding(String uniqueName) throws
        ForwardingConfigurationException {
        if (!localForwardings.containsKey(uniqueName)) {
            throw new ForwardingConfigurationException(
                "The name is not a valid forwarding configuration");
        }

        try {
            ForwardingListener listener = (ForwardingListener) localForwardings.
                get(uniqueName);

            listener.start();
        }
        catch (IOException ex) {
            throw new ForwardingConfigurationException(ex.getMessage());
        }
    }

    public void startX11Forwarding() throws TransportProtocolException,
        ServiceOperationException,
        ForwardingConfigurationException {
        if (x11ForwardingConfiguration == null) {
            throw new ForwardingConfigurationException(
                "X11 forwarding hasn't been enabled.");
        }

        try {
            ByteArrayWriter baw = new ByteArrayWriter();
            baw.writeString(x11ForwardingConfiguration.getAddressToBind());
            baw.writeInt(x11ForwardingConfiguration.getPortToBind());
            x11ForwardingConfiguration.getState().setValue(StartStopState.STARTED);
            if (log.isDebugEnabled()) {
                log.info("X11 forwarding started");
                log.debug("Address to bind: " + x11ForwardingConfiguration.getAddressToBind());
                log.debug("Port to bind: "
                              + String.valueOf(x11ForwardingConfiguration.getPortToBind()));
                log.debug("Host to connect: " + x11ForwardingConfiguration.hostToConnect);
                log.debug("Port to connect: " + x11ForwardingConfiguration.portToConnect);
            }
            else
                log.info("Request for X11 rejected.");
        }
        catch (IOException ioe) {
            throw new ForwardingConfigurationException(
                "Failed to write global request data");
        }

    }

    public void startRemoteForwarding(String name) throws
        TransportProtocolException, ServiceOperationException,
        ForwardingConfigurationException {
        try {
            if (!remoteForwardings.containsKey(name)) {
                throw new ForwardingConfigurationException(
                    "The name is not a valid forwarding configuration");
            }

            ForwardingConfiguration config = (ForwardingConfiguration)
                remoteForwardings
                .get(name);

            ByteArrayWriter baw = new ByteArrayWriter();
            baw.writeString(config.getAddressToBind());
            baw.writeInt(config.getPortToBind());

            if (connection.sendGlobalRequest(
                ForwardingServer.REMOTE_FORWARD_REQUEST, true,
                baw.toByteArray())) {
                remoteForwardings.put(name, config);

                config.getState().setValue(StartStopState.STARTED);

                log.info("Remote forwarding configuration '" + name
                         + "' started");

                if (log.isDebugEnabled()) {
                    log.debug("Address to bind: " + config.getAddressToBind());
                    log.debug("Port to bind: "
                              + String.valueOf(config.getPortToBind()));
                    log.debug("Host to connect: " + config.hostToConnect);
                    log.debug("Port to connect: " + config.portToConnect);
                }
            }
        }
        catch (IOException ioe) {
            throw new ForwardingConfigurationException(
                "Failed to write global request data");
        }
    }

    public void stopLocalForwarding(String uniqueName) throws
        ForwardingConfigurationException {
        if (!localForwardings.containsKey(uniqueName)) {
            throw new ForwardingConfigurationException(
                "The name is not a valid forwarding configuration");
        }

        ForwardingListener listener = (ForwardingListener) localForwardings.get(
            uniqueName);

        listener.stop();

        log.info("Local forwarding configuration " + uniqueName + "' stopped");
    }

    public void stopRemoteForwarding(String name) throws
        TransportProtocolException, ServiceOperationException,
        ForwardingConfigurationException {
        try {
            if (!remoteForwardings.containsKey(name)) {
                throw new ForwardingConfigurationException(
                    "The remote forwarding configuration does not exist");
            }

            ForwardingConfiguration config = (ForwardingConfiguration)
                remoteForwardings
                .get(name);

            ByteArrayWriter baw = new ByteArrayWriter();
            baw.writeString(config.getAddressToBind());
            baw.writeInt(config.getPortToBind());

            if (connection.sendGlobalRequest(
                ForwardingServer.REMOTE_FORWARD_CANCEL_REQUEST, true,
                baw.toByteArray())) {
                config.getState().setValue(StartStopState.STOPPED);

                log.info("Remote forwarding configuration '" + name
                         + "' stopped");
            }
        }
        catch (IOException ioe) {
            throw new ForwardingConfigurationException(
                "Failed to write global request data");
        }
    }

    public class ClientForwardingListener
        extends ForwardingListener {
        public ClientForwardingListener(String name,
                                        ConnectionProtocol connection,
                                        String addressToBind,
                                        int portToBind, String hostToConnect,
                                        int portToConnect) {
            super(name, connection, addressToBind, portToBind, hostToConnect,
                  portToConnect);
        }

        public ForwardingChannel createChannel(String hostToConnect,
                                               int portToConnect, Socket socket) throws
            ForwardingConfigurationException {
            return createForwardingChannel(ForwardingChannel.
                                           LOCAL_FORWARDING_CHANNEL,
                                           hostToConnect, portToConnect,
                                           ( (InetSocketAddress) socket.
                                            getRemoteSocketAddress()).
                                           getAddress().getHostAddress(),
                                           ( (InetSocketAddress) socket.
                                            getRemoteSocketAddress()).getPort());
        }
    }
}