/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.forwarding;

import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelFactory;
import com.sshtools.j2ssh.connection.ConnectionProtocol;
import com.sshtools.j2ssh.connection.GlobalRequestHandler;
import com.sshtools.j2ssh.connection.GlobalRequestResponse;
import com.sshtools.j2ssh.connection.InvalidChannelException;

import com.sshtools.j2ssh.io.ByteArrayReader;

import com.sshtools.j2ssh.transport.ServiceOperationException;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.net.Socket;
import java.net.SocketPermission;
import java.net.InetSocketAddress;

import java.util.Iterator;
import java.util.List;
import java.util.Vector;


public class ForwardingServer implements ChannelFactory, GlobalRequestHandler {
    private static Logger log = Logger.getLogger(ForwardingServer.class);
    public final static String REMOTE_FORWARD_REQUEST = "tcpip-forward";
    public final static String REMOTE_FORWARD_CANCEL_REQUEST = "cancel-tcpip-forward";
    private ConnectionProtocol connection;
    private List channelTypes = new Vector();
    private List localForwardings = new Vector();
    private List remoteForwardings = new Vector();

    public ForwardingServer(ConnectionProtocol connection)
        throws IOException {
        this.connection = connection;
        channelTypes.add(ForwardingChannel.LOCAL_FORWARDING_CHANNEL);

        connection.addChannelFactory(ForwardingChannel.LOCAL_FORWARDING_CHANNEL, this);
        connection.allowGlobalRequest(REMOTE_FORWARD_REQUEST, this);
        connection.allowGlobalRequest(REMOTE_FORWARD_CANCEL_REQUEST, this);
    }

   /* public List getChannelType() {
        return channelTypes;
    }*/

    public Channel createChannel(String channelType, byte[] requestData)
        throws InvalidChannelException {


        if (!channelType.equals(ForwardingChannel.LOCAL_FORWARDING_CHANNEL)) {
            throw new InvalidChannelException(
                "The client can only request the "
                + "opening of a local forwarding channel");
        }

        try {
            ByteArrayReader bar = new ByteArrayReader(requestData);
            String hostToConnect = bar.readString();
            int portToConnect = (int) bar.readInt();
            String originatingHost = bar.readString();
            int originatingPort = (int) bar.readInt();

            // Get a configuration item for the forwarding
            ForwardingConfiguration config = getLocalForwardingByAddress(originatingHost,
                    originatingPort);

            Socket socket = new Socket(hostToConnect, portToConnect);

            // Create the channel adding it to the active channels
            ForwardingChannel channel = config.createForwardingChannel(channelType,
                    hostToConnect, portToConnect, originatingHost, originatingPort);

            channel.bindSocket(socket);

            return channel;
        } catch (ForwardingConfigurationException fce) {
            throw new InvalidChannelException(
                "No valid forwarding configuration was available for the request");
              } catch (IOException ioe) {
                  throw new InvalidChannelException("The channel request data is "
                      + "invalid/or corrupt for channel type " + channelType);
              }



    }

    public GlobalRequestResponse processGlobalRequest(String requestName,
        byte[] requestData) {
        GlobalRequestResponse response = GlobalRequestResponse.REQUEST_FAILED;
        String addressToBind = null;
        int portToBind = -1;

        log.debug("Processing " + requestName + " global request");

        try {
            ByteArrayReader bar = new ByteArrayReader(requestData);
            addressToBind = bar.readString();
            portToBind = (int) bar.readInt();

            if (requestName.equals(REMOTE_FORWARD_REQUEST)) {
                addRemoteForwardingConfiguration(addressToBind, portToBind);
                response = GlobalRequestResponse.REQUEST_SUCCEEDED;
            }

            if (requestName.equals(REMOTE_FORWARD_CANCEL_REQUEST)) {
                removeRemoteForwarding(addressToBind, portToBind);
                response = GlobalRequestResponse.REQUEST_SUCCEEDED;
            }
        } catch (IOException ioe) {
            log.warn("The client failed to request " + requestName + " for "
                + addressToBind + ":" + String.valueOf(portToBind), ioe);
        }

        return response;
    }

    protected ForwardingConfiguration getLocalForwardingByAddress(
        String orginatingAddress, int originatingPort)
        throws ForwardingConfigurationException {

        try {
          Iterator it = localForwardings.iterator();
          ForwardingConfiguration config;

          while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(orginatingAddress)
                && (config.getPortToBind() == originatingPort)) {
              return config;
            }
          }

          // No configuration is available so create one
          config = new ForwardingConfiguration(orginatingAddress,
                                               originatingPort);

          // We must start this configuration in order to use it
          config.start();

          localForwardings.add(config);

          return config;
        }
        catch (IOException ex) {
          throw new ForwardingConfigurationException(ex.getMessage());
        }
    }

    protected ForwardingConfiguration getRemoteForwardingByAddress(
        String addressToBind, int portToBind)
        throws ForwardingConfigurationException {
        Iterator it = remoteForwardings.iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                    && (config.getPortToBind() == portToBind)) {
                return config;
            }
        }

        throw new ForwardingConfigurationException(
            "The remote forwarding does not exist!");
    }

    protected void addRemoteForwardingConfiguration(String addressToBind,
        int portToBind) throws ForwardingConfigurationException {
        // Is the server already listening
        Iterator it = remoteForwardings.iterator();
        ForwardingConfiguration config;

        while (it.hasNext()) {
            config = (ForwardingConfiguration) it.next();

            if (config.getAddressToBind().equals(addressToBind)
                    && (config.getPortToBind() == portToBind)) {
                throw new ForwardingConfigurationException(
                    "The address and port are already in use!");
            }
        }

        config = new ForwardingConfiguration(addressToBind, portToBind);

        // Check the security mananger
        SecurityManager manager = System.getSecurityManager();

        if (manager != null) {
            try {
                manager.checkPermission(new SocketPermission(addressToBind
                        + ":" + String.valueOf(portToBind), "accept,listen"));
            } catch (SecurityException e) {
                throw new ForwardingConfigurationException(
                    "The security manager has denied listen permision on "
                    + addressToBind + ":" + String.valueOf(portToBind));
            }
        }

        try {
          ForwardingListener listener = new ServerForwardingListener(connection,
              addressToBind, portToBind);
          remoteForwardings.add(listener);
          listener.start();
        }
        catch (IOException ex) {
          throw new ForwardingConfigurationException(ex.getMessage());
        }
    }

    protected void removeRemoteForwarding(String addressToBind, int portToBind)
        throws ForwardingConfigurationException {
        ForwardingConfiguration config = getRemoteForwardingByAddress(addressToBind,
                portToBind);

        // Stop the forwarding
        config.stop();

        // Remove from the remote forwardings list
        remoteForwardings.remove(config);
    }

    class ServerForwardingListener extends ForwardingListener {
        public ServerForwardingListener(ConnectionProtocol connection,
            String addressToBind, int portToBind) {
            super(connection, addressToBind, portToBind);
        }

        public ForwardingChannel createChannel(String hostToConnect,
            int portToConnect, Socket socket)
            throws ForwardingConfigurationException {

            try {
              ForwardingChannel channel = createForwardingChannel(
                  ForwardingChannel.REMOTE_FORWARDING_CHANNEL,
                  hostToConnect, portToConnect,
                  ( (InetSocketAddress) socket.getRemoteSocketAddress()).
                  getAddress().getHostAddress(),
                  ( (InetSocketAddress) socket.getRemoteSocketAddress()).
                  getPort());

              channel.bindSocket(socket);

              return channel;
            }
            catch (IOException ex) {
              throw new ForwardingConfigurationException(ex.getMessage());
            }
        }
    }
}
