/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.session;

import java.io.IOException;
import java.io.InputStream;

import org.apache.log4j.Logger;
import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelInputStream;
import com.sshtools.j2ssh.connection.ChannelFactory;
import com.sshtools.j2ssh.connection.InvalidChannelException;
import com.sshtools.j2ssh.connection.SshMsgChannelExtendedData;
import com.sshtools.j2ssh.io.ByteArrayReader;
import com.sshtools.j2ssh.io.ByteArrayWriter;
import com.sshtools.j2ssh.io.UnsignedInteger32;
import com.sshtools.j2ssh.subsystem.SubsystemClient;
import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.TransportProtocolException;
import com.sshtools.j2ssh.agent.AgentForwardingChannel;
import com.sshtools.j2ssh.agent.SshAgentClient;
import com.sshtools.j2ssh.agent.*;
import java.net.Socket;

public class SessionChannelClient
    extends Channel {
  private static Logger log = Logger.getLogger(SessionChannelClient.class);
  private Integer exitCode = null;
  private ChannelInputStream stderr = null;
  private String sessionType = "Uninitialized";
  private SubsystemClient subsystem;
  private boolean localFlowControl = false;
  private SignalListener signalListener;

  public SessionChannelClient() {
    super();
  }

  public byte[] getChannelOpenData() {
    return null;
  }

  public byte[] getChannelConfirmationData() {
    return null;
  }

  public String getChannelType() {
    return "session";
  }

  protected int getMinimumWindowSpace() {
    return 1024;
  }

  protected int getMaximumWindowSpace() {
    return 32648;
  }

  protected int getMaximumPacketSize() {
    return 32648;
  }

  public void setSignalListener(SignalListener signalListener) {
    this.signalListener = signalListener;
  }

  public boolean setEnvironmentVariable(String name, String value) throws
      TransportProtocolException, ServiceOperationException {
    log.debug("Requesting environment variable to be set [" + name + "="
              + value + "]");

    try {
      ByteArrayWriter baw = new ByteArrayWriter();

      baw.writeString(name);
      baw.writeString(value);

      return connection.sendChannelRequest(this, "env", true,
                                           baw.toByteArray());
    }
    catch (IOException ioe) {
      throw new ServiceOperationException(
          "Could not set environment variable;"
          + " error writing request data");
    }
  }

  public boolean requestAgentForwarding() throws IOException {

    log.info("Requesting agent forwarding for the session");

    boolean success = connection.sendChannelRequest(
        this,
        "auth-agent-req",
        true,
        null);

    if(success) {
      // Allow an Agent Channel to be opened
      connection.addChannelFactory(AgentForwardingChannel.AGENT_FORWARDING_CHANNEL,
                                  new ChannelFactory() {
          public Channel createChannel(String channelType, byte[] requestData)
                  throws InvalidChannelException {

                try {
                  AgentForwardingChannel channel = new AgentForwardingChannel(false);

                  Socket socket = SshAgentClient.connectAgentSocket(System.getProperty("sshtools.agent"), 5);

                  channel.bindSocket(socket);
                  return channel;
                }
                catch (Exception ex) {
                  throw new InvalidChannelException(ex.getMessage());
                }
          }
         });
    }

    return success;

  }

  public boolean requestX11Forwarding(int display, String cookie) throws
      TransportProtocolException, ServiceOperationException {
    log.debug("Requesting X11 forwarding for display " +
              display + " using cookie " + cookie);
    try {
      ByteArrayWriter baw = new ByteArrayWriter();
      baw.writeBoolean(false);
      baw.writeString("MIT-MAGIC-COOKIE-1");
      baw.writeString(cookie);
      baw.writeUINT32(new UnsignedInteger32(String.valueOf(display)));
      return connection.sendChannelRequest(this, "x11-req", true,
                                           baw.toByteArray());
    }
    catch (IOException ioe) {
      throw new ServiceOperationException(
          "Could not set environment variable;"
          + " error writing request data");
    }
  }

  public Integer getExitCode() {
    return exitCode;
  }

  public void changeTerminalDimensions(PseudoTerminal term) throws
      TransportProtocolException, ServiceOperationException {
    log.debug("Changing terminal dimensions");

    try {
      ByteArrayWriter baw = new ByteArrayWriter();

      baw.writeInt(term.getColumns());
      baw.writeInt(term.getRows());
      baw.writeInt(term.getWidth());
      baw.writeInt(term.getHeight());

      connection.sendChannelRequest(this, "window-change", false,
                                    baw.toByteArray());
    }
    catch (IOException ioe) {
      throw new ServiceOperationException(
          "Failed to change terminal dimensions; error writing request data");
    }
  }

  public boolean executeCommand(String command) throws
      TransportProtocolException, ServiceOperationException {
    log.info("Requesting command execution");
    log.debug("Command is " + command);

    try {
      ByteArrayWriter baw = new ByteArrayWriter();

      baw.writeString(command);

      if (connection.sendChannelRequest(this, "exec", true,
                                        baw.toByteArray())) {
        if (sessionType.equals("Uninitialized")) {
          sessionType = "exec " + command;
        }
        return true;
      }
      else {
        return false;
      }
    }
    catch (IOException ioe) {
      throw new ServiceOperationException(
          "Failed to request command execution; error writing request data");
    }
  }

  public boolean requestPseudoTerminal(PseudoTerminal term) throws
      TransportProtocolException, ServiceOperationException {
    log.info("Requesting pseudo terminal");
    log.debug("Terminal Type is " + term.getTerm());

    // This requests a pseudo terminal
    try {
      ByteArrayWriter baw = new ByteArrayWriter();
      baw.writeString(term.getTerm());
      baw.writeInt(term.getColumns());
      baw.writeInt(term.getRows());
      baw.writeInt(term.getWidth());
      baw.writeInt(term.getHeight());
      baw.writeString(term.getEncodedTerminalModes());

      return connection.sendChannelRequest(this, "pty-req", true,
                                           baw.toByteArray());
    }
    catch (IOException ioe) {
      throw new ServiceOperationException(
          "Failed to write channel request data to ByteArrayWriter");
    }
  }

  public boolean startShell() throws IOException {
    log.debug("Requesting users shell");

    // Send the request for a shell, we want a reply
    if (connection.sendChannelRequest(this, "shell", true, null)) {
      if (sessionType.equals("Uninitialized")) {
        sessionType = "shell";
      }
      return true;
    }
    else {
      return false;
    }

  }

  public boolean startSubsystem(String subsystem) throws
      TransportProtocolException, ServiceOperationException {
    log.info("Starting " + subsystem + " subsystem");

    try {
      ByteArrayWriter baw = new ByteArrayWriter();

      baw.writeString(subsystem);

      if (connection.sendChannelRequest(this, "subsystem", true,
                                        baw.toByteArray())) {
        if (sessionType.equals("Uninitialized")) {
          sessionType = subsystem;
        }
        return true;
      }
      else {
        return false;
      }
    }
    catch (IOException ioe) {
      throw new ServiceOperationException("Failed to start the "
                                          + subsystem +
          " subsystem; could not write request data");
    }
  }

  public boolean startSubsystem(SubsystemClient subsystem) throws IOException {
    boolean result = startSubsystem(subsystem.getName());

    if (result) {
      this.subsystem = subsystem;
      subsystem.setSessionChannel(this);
      subsystem.start();
    }

    return result;
  }

  public boolean isLocalFlowControlEnabled() {
    return localFlowControl;
  }

  public String getSessionType() {
    return sessionType;
  }

  public void setSessionType(String sessionType) {
    this.sessionType = sessionType;
  }

  public SubsystemClient getSubsystem() {
    return subsystem;
  }

  protected void onChannelClose() throws ServiceOperationException {
    Integer exitCode = getExitCode();

    if (exitCode != null) {
      log.debug("Exit code " + exitCode.toString());
    }
  }

  protected void onChannelEOF() throws ServiceOperationException {
  }

  protected void onChannelOpen() throws InvalidChannelException {
    stderr = new ChannelInputStream(incoming,
                                    new Integer(SshMsgChannelExtendedData.
                                                SSH_EXTENDED_DATA_STDERR));
  }

  public InputStream getStderrInputStream() throws ServiceOperationException {
    if (stderr == null) {
      throw new ServiceOperationException(
          "The session must be started first!");
    }

    return stderr;
  }

  protected void onChannelRequest(String requestType, boolean wantReply,
                                  byte[] requestData) throws IOException {
    log.debug("Channel Request received: " + requestType);


    if (requestType.equals("exit-status")) {

      exitCode = new Integer((int)ByteArrayReader.readInt(requestData, 0));
      log.debug("Exit code of " + exitCode.toString() + " received");


    }
    else if (requestType.equals("exit-signal")) {

      ByteArrayReader bar = new ByteArrayReader(requestData);
      String   signal = bar.readString();
      boolean  coredump = bar.read()!=0;
      String   message = bar.readString();
      String   language = bar.readString();

      log.debug("Exit signal " + signal + " received");
      log.debug("Signal message: " + message);
      log.debug("Core dumped: " + String.valueOf(coredump));

      if(signalListener!=null)
        signalListener.onExitSignal(signal, coredump, message);

      } else if(requestType.equals("xon-xoff")) {

      if(requestData.length >= 1)
        localFlowControl = (requestData[0]!=0);

    } else if(requestType.equals("signal")) {

      String signal = ByteArrayReader.readString(requestData, 0);
      log.debug("Signal " + signal + " received");

      if(signalListener!=null)
        signalListener.onSignal(signal);

    }
    else {
      if (wantReply) {
        connection.sendChannelRequestFailure(this);
      }
    }

  }
}
