/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport;

import com.sshtools.j2ssh.SshException;
import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.PlatformConfiguration;

import com.sshtools.j2ssh.util.InvalidStateException;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


public abstract class AsyncService extends Service implements Runnable {
    private static Logger log = Logger.getLogger(Service.class);
    protected SshThread thread;

    public AsyncService(String serviceName) {
        super(serviceName);
    }

    protected void onStart() throws IOException {
        if (Thread.currentThread() instanceof SshThread) {
            thread = ((SshThread) Thread.currentThread()).cloneThread(this,
                    getServiceName());
        } else {
            thread = new SshThread(this, getServiceName(), true);
        }

        log.info("Starting " + getServiceName() + " service thread");
        thread.start();
    }

    public final void run() {
        int[] messageFilter = getAsyncMessageFilter();

        state.setValue(ServiceState.SERVICE_STARTED);

        while (state.getValue() == ServiceState.SERVICE_STARTED) {
            try {
                // Get the next message from the message store
                SshMessage msg = messageStore.getMessage(messageFilter);

                if (state.getValue() == ServiceState.SERVICE_STOPPED) {
                    break;
                }

                log.info(getServiceName() + " is processing "
                    + msg.getMessageName());
                onMessageReceived(msg);
            } catch (MessageStoreEOFException ioe) {
                log.debug("The message store is EOF");
                stop();
            } catch(IOException ioe) {
              if (state.getValue() != ServiceState.SERVICE_STOPPED) {
                    log.fatal("Service message loop failed!", ioe);
                    stop();
                }
            }
        }

        onStop();
        log.info(getServiceName() + " thread is exiting");
        thread = null;
    }

    protected abstract void onStop();

    protected abstract int[] getAsyncMessageFilter();

    protected abstract void onMessageReceived(SshMessage msg)
        throws IOException;
}
