/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport.kex;

import com.sshtools.j2ssh.SshException;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;

import com.sshtools.j2ssh.transport.MessageAlreadyRegisteredException;
import com.sshtools.j2ssh.transport.TransportProtocol;

import com.sshtools.j2ssh.transport.publickey.SshPrivateKey;

import com.sshtools.j2ssh.util.Hash;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.math.BigInteger;

import java.security.NoSuchAlgorithmException;

import java.util.Random;


public class DhGroup1Sha1 extends SshKeyExchange {
    private static Logger log = Logger.getLogger(DhGroup1Sha1.class);
    private BigInteger e = null;
    private BigInteger f = null;
    private BigInteger g = new BigInteger("2");
    private BigInteger p = new BigInteger("FFFFFFFFFFFFFFFFC90FDAA22168C234"
            + "C4C6628B80DC1CD129024E088A67CC74"
            + "020BBEA63B139B22514A08798E3404DD"
            + "EF9519B3CD3A431B302B0A6DF25F1437"
            + "4FE1356D6D51C245E485B576625E7EC6"
            + "F44C42E9A637ED6B0BFF5CB6F406B7ED"
            + "EE386BFB5A899FA5AE9F24117C4B1FE6"
            + "49286651ECE65381FFFFFFFFFFFFFFFF", 16);
    private BigInteger q = new BigInteger("7FFFFFFFFFFFFFFFE487ED5110B4611A"
            + "62633145C06E0E68948127044533E63A"
            + "0105DF531D89CD9128A5043CC71A026E"
            + "F7CA8CD9E69D218D98158536F92F8A1B"
            + "A7F09AB6B6A8E122F242DABB312F3F63"
            + "7A262174D31BF6B585FFAE5B7A035BF6"
            + "F71C35FDAD44CFD2D74F9208BE258FF3"
            + "24943323F67329C0FFFFFFFFFFFFFFFF", 16);
    private BigInteger x = null;
    private BigInteger y = null;
    private String clientId;
    private String serverId;
    private byte[] clientKexInit;
    private byte[] serverKexInit;

    public DhGroup1Sha1() {
    }

    protected void onInit() {
        messageStore.registerMessage(SshMsgKexDhInit.SSH_MSG_KEXDH_INIT,
            SshMsgKexDhInit.class);

        messageStore.registerMessage(SshMsgKexDhReply.SSH_MSG_KEXDH_REPLY,
            SshMsgKexDhReply.class);
    }

    public void performClientExchange(String clientId, String serverId,
        byte[] clientKexInit, byte[] serverKexInit) throws IOException {
        log.info("Starting client side key exchange.");

        this.clientId = clientId;
        this.serverId = serverId;
        this.clientKexInit = clientKexInit;
        this.serverKexInit = serverKexInit;

        int minBits = g.bitLength();
        int maxBits = q.bitLength();

        Random rnd = ConfigurationLoader.getRND();

        // Generate a random bit count for the random x value
        int genBits = (int) (((maxBits - minBits + 1) * rnd.nextFloat())
            + minBits);

        x = new BigInteger(genBits, rnd);

        // Calculate e
        e = g.modPow(x, p);

        // Prepare the message
        SshMsgKexDhInit msg = new SshMsgKexDhInit(e);

        // Send it
        try {
            transport.sendMessage(msg, this);
        } catch (SshException tpe) {
            throw new KeyExchangeException(
                "Failed to send key exchange initailaztion message");
        }

        int[] messageId = new int[1];
        messageId[0] = SshMsgKexDhReply.SSH_MSG_KEXDH_REPLY;

        SshMsgKexDhReply reply = (SshMsgKexDhReply) transport.readMessage(messageId);

        hostKey = reply.getHostKey();
        signature = reply.getSignature();

        f = reply.getF();

        // Calculate diffe hellman k value
        secret = f.modPow(x, p);

        // Calculate the exchange hash
        calculateExchangeHash();
    }

    public void performServerExchange(String clientId, String serverId,
        byte[] clientKexInit, byte[] serverKexInit, SshPrivateKey prvKey)
        throws IOException {
        try {
            this.clientId = clientId;
            this.serverId = serverId;
            this.clientKexInit = clientKexInit;
            this.serverKexInit = serverKexInit;

            int minBits = g.bitLength();
            int maxBits = q.bitLength();

            Random rnd = ConfigurationLoader.getRND();

            // Generate a random bit count for the random x value
            int genBits = (int) (((maxBits - minBits + 1) * rnd.nextFloat())
                + minBits);

            y = new BigInteger(genBits, rnd);

            // Calculate f
            f = g.modPow(y, p);

            // Wait for the e value and calculate the other parameters
            int[] messageId = new int[1];
            messageId[0] = SshMsgKexDhInit.SSH_MSG_KEXDH_INIT;

            SshMsgKexDhInit msg = (SshMsgKexDhInit) transport.readMessage(messageId);

            e = msg.getE();

            // Calculate k
            secret = e.modPow(y, p);

            hostKey = prvKey.getPublicKey().getEncoded();

            calculateExchangeHash();

            signature = prvKey.generateSignature(exchangeHash);

            SshMsgKexDhReply reply = new SshMsgKexDhReply(hostKey, f, signature);

            transport.sendMessage(reply, this);
        } catch (SshException e) {
            throw new KeyExchangeException(e.getMessage());
        }
    }

    protected void calculateExchangeHash() throws KeyExchangeException {
        Hash hash;

        try {
            // Start a SHA hash
            hash = new Hash("SHA");
        } catch (NoSuchAlgorithmException nsae) {
            throw new KeyExchangeException("SHA algorithm not supported");
        }

        int i;

        // The local software version comments
        hash.putString(clientId);

        // The remote software version comments
        hash.putString(serverId);

        // The local kex init payload
        hash.putInt(clientKexInit.length);
        hash.putBytes(clientKexInit);

        // The remote kex init payload
        hash.putInt(serverKexInit.length);
        hash.putBytes(serverKexInit);

        // The host key
        hash.putInt(hostKey.length);
        hash.putBytes(hostKey);

        // The diffie hellman e value
        hash.putBigInteger(e);

        // The diffie hellman f value
        hash.putBigInteger(f);

        // The diffie hellman k value
        hash.putBigInteger(secret);

        // Do the final output
        exchangeHash = hash.doFinal();
    }
}
