/*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh.transport.publickey;

import com.sshtools.j2ssh.io.ByteArrayReader;

import com.sshtools.j2ssh.util.Base64;

import java.util.StringTokenizer;


public class OpenSSHPublicKeyFormat implements SshPublicKeyFormat {
    String comment = null;

    public OpenSSHPublicKeyFormat(String comment) {
        setComment(comment);
    }

    public OpenSSHPublicKeyFormat() {
    }

    public void setComment(String comment) {
        this.comment = comment;
    }

    public String getComment() {
        return comment;
    }

    public String getFormatType() {
        return "OpenSSH-PublicKey";
    }

    public byte[] getKeyBlob(byte[] formattedKey) throws InvalidSshKeyException {
        String temp = new String(formattedKey);

        // Get the algorithm name end index
        int i = temp.indexOf(" ");

        if (i > 0) {
            String algorithm = temp.substring(0, i);

            if (supportsAlgorithm(algorithm)) {
                // Get the keyblob end index
                int i2 = temp.indexOf(" ", i + 1);

                if (i2 > i) {
                    String encoded = temp.substring(i + 1, i2);

                    if (temp.length() > i2) {
                        comment = temp.substring(i2 + 1).trim();
                    }

                    return Base64.decode(encoded);
                }
            }
        }

        throw new InvalidSshKeyException("Failed to read OpenSSH key format");
    }

    public byte[] formatKey(byte[] keyblob) {
        String algorithm = ByteArrayReader.readString(keyblob, 0);

        String formatted = algorithm + " " + Base64.encodeBytes(keyblob, true);

        if (comment != null) {
            formatted += (" " + comment);
        }

        return formatted.getBytes();
    }

    public boolean isFormatted(byte[] formattedKey) {
        String test = new String(formattedKey).trim();

        if (test.startsWith("ssh-dss") || test.startsWith("ssh-rsa")) {
            return true;
        } else {
            return false;
        }
    }

    public boolean supportsAlgorithm(String algorithm) {
        if (algorithm.equals("ssh-dss") || algorithm.equals("ssh-rsa")) {
            return true;
        } else {
            return false;
        }
    }
}
