/* Copyright (c) 1988 Jerry Joplin
 *
 * Portions copyright (c) 1981, 1988
 * Trustees of Columbia University in the City of New York
 *
 * Permission is granted to any individual or institution
 * to use, copy, or redistribute this program and
 * documentation as long as it is not sold for profit and
 * as long as the Columbia copyright notice is retained.
 *
 *
 * Modified by Nagy Daniel - tailored for DJGPP and Linux terminal
 */

#include <stdio.h>
#include <dos.h>
#include "include/protocol.h"
#ifdef __TURBOC__
#include <bios.h>
#elif __DJGPP__
#include <dpmi.h>
#endif

#define ESC         0x1B	/* ASCII ESCape character */
#define ESCAPE      0x11B	/* Keyboard ESCape scan code */
#define DEL         0x7F	/* ASCII DELete character */
#define BKSP        0xE08	/* Keyboard BacKSPace scan code */
#define F1          0x3B00	/* Keyboard Function key 1 scan code */
#define F2          0x3C00	/* Keyboard Function key 2 scan code */
#define F3          0x3D00	/* Keyboard Function key 3 scan code */
#define F4          0x3E00	/* Keyboard Function key 4 scan code */
#define F5          0x3F00	/* Keyboard Function key 5 scan code */
#define F6          0x4000	/* Keyboard Function key 6 scan code */
#define F7          0x4100	/* Keyboard Function key 7 scan code */
#define F8          0x4200	/* Keyboard Function key 8 scan code */
#define F9          0x4300	/* Keyboard Function key 9 scan code */
#define F10         0x4400	/* Keyboard Function key 10 scan code */
#define UP          0x4800	/* Keyboard Up Arrow scan code */
#define DOWN        0x5000	/* Keyboard Down Arrow scan code */
#define LEFT        0x4B00	/* Keyboard Left Arrow scan code */
#define RIGHT       0x4D00	/* Keyboard Right Arrow scan code */
#define K7          0x4737	/* Keyboard Numeric 7 scan code */
#define K8          0x4838	/* Keyboard Numeric 8 scan code */
#define K9          0x4939	/* Keyboard Numeric 9 scan code */
#define KDASH       0x372A	/* Keyboard Numeric Asteric scan code */
#define K4          0x4B34	/* Keyboard Numeric 4 scan code */
#define K5          0x4C35	/* Keyboard Numeric 5 scan code */
#define K6          0x4D36	/* Keyboard Numeric 6 scan code */
#define KCOMA       0x4A2D	/* Keyboard Numeric Dash(minus) scan code */
#define K1          0x4F31	/* Keyboard Numeric 1 scan code */
#define K2          0x5032	/* Keyboard Numeric 2 scan code */
#define K3          0x5133	/* Keyboard Numeric 3 scan code */
#define KENTR       0x4E2B	/* Keyboard Numeric + (plus) scan code */
#define K0          0x5230	/* Keyboard Numeric 0 scan code */
#define KDOT        0x532E	/* Keyboard Numeric Period scan code */
#define INSERT	    0x5200	/* Gray insert */
#define DELETE	    0x5300	/* Gray delete */
#define HOME	    0x4700	/* Gray home */
#define END	    0x4F00	/* Gray end */
#define PGUP	    0x4900	/* Gray page up */
#define PGDOWN	    0x5100	/* Gray page down */


/*****************************************************************************/
/* function prototypes                                                       */

void KeyInit(void);		/* Initialize the keyboard system */
int ConChk(void);		/* Check the keyboard for keystrokes */
unsigned int GetKey(void);	/* Retrieve a scan code from keyboard */
int DoKey(void);		/* Transmit a key sequence */

static void TransKey(unsigned);	/* Translate a keystroke to a sequence */
static int TransNumKey(unsigned);	/* Translate Numeric Keypad keystroke */
static int TransApplKey(unsigned);	/* Translate Application Keypad keystroke */
static void SendBksp(void);	/* Transmit the backspace character */
void SetKeyPad(int);		/* Set the keypad to APPLICATION, NUMERIC */
void SetCursorKey(int);		/* Set the cursor key mode */

/*****************************************************************************/
/* Global variables                                                          */

unsigned char backspace;	/* Backspace/Delete translation flag */
unsigned char keyclick;		/* Keyclick mode on/off */
unsigned char applkeypad;	/* Current state of keypad */

void TTSetup(void);		/* Communications setup function */
void VidSetup(void);		/* Video setup function */
void KeySetup(void);		/* Keyboard setup function */
void VTSetup(void);		/* VT emulation setup function */
void FileSetup(void);		/* File system setup function */

/*****************************************************************************/
/* External variables                                                        */

/*****************************************************************************/
/* Local Static data                                                         */

static char cursorkey = '[';	/* Sequence character in cursor key */
#ifdef __TURBOC__
static union REGS regs;		/* Registers for int86 call */
#elif __DJGPP__
__dpmi_regs regs;
#endif

/*****************************************************************************/
/*****************************************************************************/

/* K E Y I N I T -- Initialize the keyboard system */

void KeyInit(void)
{
	delay(1);		/* Initialize Turbo C delay function */
	backspace = 0;
	keyclick = 0;
	applkeypad = 0;
}

/*  C O N C H K  --  Check if any key strokes are waiting, check hot keys */

int ConChk(void)
{
	void (*setupfunct) (void);	/* Pointer to selected setup function */

	regs.h.ah = 0x1;		/* Use function 1 of interrupt 0x16 */

#ifdef __TURBOC__

	int86(0x16, &regs, &regs);	/* to check for waiting keystrokes */

#elif __DJGPP__

	__dpmi_int(0x16, &regs);	/* to check for waiting keystrokes */

#endif

	/* If the zero flag is set then no keys */
	return (regs.x.flags & 0x40 ? 0 : 1);
}

/*  G E T K E Y  --  Return a keyboard scan code */

unsigned int GetKey(void)
{
	register unsigned int scancode;

	scancode = bioskey(0);	/* Get a keystroke, waits if none ready */
	if (keyclick) {		/* If keyclick flag is set */
		sound(250);	/* Turn on low frequency sound */
		delay(5);	/* Wait a short time period */
		nosound();	/* Turn off the sound */
	}
	return (scancode);	/* Return the retrieved scancode */
}

/*  D O K E Y  --  Retrieve and interpret a keystroke */

int DoKey(void)
{
	unsigned scancode;

	scancode = GetKey();	/* Get a keystroke, waits if none ready */
	TransKey(scancode);
	return (0);		/* return success */
}

/* T R A N S K E Y  -- translate a scancode into a keystroke sequence */

static void TransKey(unsigned key)
{
	switch (key) {		/* Evaluate this keyboard scan code */
	case BKSP:		/* Backspace pressed */
		SendBksp();
		break;
	case F1:		/* Function key 1 pressed */
		ttoea2c(cursorkey, 'A');
		break;
	case F2:		/* Function key 2 pressed */
		ttoea2c(cursorkey, 'B');
		break;
	case F3:		/* Function key 3 pressed */
		ttoea2c(cursorkey, 'C');
		break;
	case F4:		/* Function key 4 pressed */
		ttoea2c(cursorkey, 'D');
		break;
	case F5:		/* Function key 5 pressed */
		ttoea2c(cursorkey, 'E');
		break;
	case F6:		/* Function key 6 pressed */
		ttoea3c('1','7',0x7e);
		break;
	case F7:		/* Function key 7 pressed */
		ttoea3c('1','8',0x7e);
		break;
	case F8:		/* Function key 8 pressed */
		ttoea3c('1','9',0x7e);
		break;
	case F9:		/* Function key 9 pressed */
		ttoea3c('2','0',0x7e);
		break;
	case F10:		/* Function key 10 pressed */
		ttoea3c('2','1',0x7e);
		break;
	case UP:		/* Up Arrow pressed */
		ttoe2c(cursorkey, 'A');
		break;
	case DOWN:		/* Down Arrow pressed */
		ttoe2c(cursorkey, 'B');
		break;
	case RIGHT:		/* Right Arrow pressed */
		ttoe2c(cursorkey, 'C');
		break;
	case LEFT:		/* Left Arrow pressed */
		ttoe2c(cursorkey, 'D');
		break;
        case INSERT:
                ttoea2c('2', 0x7E);
                break;
        case DELETE:
                ttoea2c('3', 0x7E);
                break;
        case HOME:
                ttoea2c('1', 0x7E);
                break;
        case END:
                ttoea2c('4', 0x7E);
                break;
        case PGUP:
                ttoea2c('5', 0x7E);
                break;
        case PGDOWN:
                ttoea2c('6', 0x7E);
                break;

	default:		/* No translation yet, check numeric pad */
		if ((TransNumKey(key) == 0) && (TransApplKey(key) == 0))
			ttoc((char) key);	/* Still no translation, transmit char */
		break;
	}
}

/* T R A N S N U M K E Y  --  Try and translate key from the Numeric Keypad */

static TransNumKey(register unsigned key)
{

	if (applkeypad != 0)	/* If keypad is not in NUMERIC mode */
		return (0);	/* then no translation here possible */

	switch (key) {
	case K7:		/* Numeric 7 pressed */
		ttoc('7');
		break;
	case K8:		/* Numeric 8 pressed */
		ttoc('8');
		break;
	case K9:		/* Numeric 9 pressed */
		ttoc('9');
		break;
	case KDASH:		/* Numeric Minus pressed */
		ttoc('-');
		break;
	case K4:		/* Numeric 4 pressed */
		ttoc('4');
		break;
	case K5:		/* Numeric 5 pressed */
		ttoc('5');
		break;
	case K6:		/* Numeric 6 pressed */
		ttoc('6');
		break;
	case KCOMA:		/* Numeric Comma pressed */
		ttoc(',');
		break;
	case K1:		/* Numeric 1 pressed */
		ttoc('1');
		break;
	case K2:		/* Numeric 2 pressed */
		ttoc('2');
		break;
	case K3:		/* Numeric 3 pressed */
		ttoc('3');
		break;
	case K0:		/* Numeric 0 pressed */
		ttoc('0');
		break;
	case KDOT:		/* Numeric Period pressed */
		ttoc('.');
		break;
	case KENTR:		/* Numeric Enter pressed */
		ttoc(13);
		break;
	default:
		return (0);	/* No translation */
	}
	return (1);
}

/* T R A N S A P P L K E Y  --  Try and translate key from Application Keypad*/

static TransApplKey(register unsigned key)
{

	if (applkeypad != 1)	/* If keypad is not APPLICATION mode */
		return (0);	/* then no translation here possible */

	switch (key) {
	case K7:		/* Application key 7 pressed */
		ttoe2c('O', 'w');
		break;
	case K8:		/* Application key 8 pressed */
		ttoe2c('O', 'x');
		break;
	case K9:		/* Application key 9 pressed */
		ttoe2c('O', 'y');
		break;
	case KDASH:		/* Application key minus pressed */
		ttoe2c('O', 'm');
		break;
	case K4:		/* Application key 4 pressed */
		ttoe2c('O', 't');
		break;
	case K5:		/* Application key 5 pressed */
		ttoe2c('O', 'u');
		break;
	case K6:		/* Application key 6 pressed */
		ttoe2c('O', 'v');
		break;
	case KCOMA:		/* Application key Comma pressed */
		ttoe2c('O', 'l');
		break;
	case K1:		/* Application key 1 pressed */
		ttoe2c('O', 'q');
		break;
	case K2:		/* Application key 2 pressed */
		ttoe2c('O', 'r');
		break;
	case K3:		/* Application key 3 pressed */
		ttoe2c('O', 's');
		break;
	case K0:		/* Application key 0 pressed */
		ttoe2c('O', 'p');
		break;
	case KDOT:		/* Application key Dot pressed */
		ttoe2c('O', 'n');
		break;
	case KENTR:		/* Application key Enter pressed */
		ttoe2c('O', 'M');
		break;
	default:
		return (0);	/* No translation */
	}
	return (1);
}

/* S E N D B K S P -- Send a backspace out */

static void SendBksp(void)
{
	ttoc (backspace ? 8 : DEL);
}

/* S E T K E Y P A D -- Set the keypad translation */

void SetKeyPad(int mode)
{
	applkeypad = mode ? 1 : 0;	/* keypad = APPLICATION/NUMERIC */
}

/* S E T C U R S O R K E Y -- Set the cursior key mode */

void SetCursorKey(mode)
{	
	/* This establishes the second character */
	/* of the cursor keys escape sequence */
	cursorkey = mode ? 'O' : '[';
}
