/* negotiat.c       Copyright (c) 2000 Nagy Daniel
 *
 * $Date: 2000/12/06 12:50:30 $
 * $Revision: 1.4 $
 *
 * This module is the SSH negotiation part:
 *  - open TCP connection
 *  - version check
 *  - key generation
 *  - user authorization
 *  - TTY allocation
 *  - interactive shell request
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include<io.h>
#include<stdio.h>
#include<stdlib.h>
#include<conio.h>
#include<string.h>

#ifdef __DJGPP__
#include<errno.h>
#include"include/tcp_djgp.h"
#elif __TURBOC__
#include"include/tcp.h"
#endif

#include"include/ssh.h"
#include"include/md5.h"
#include"include/rsa.h"
#include"include/des.h"
#include"include/blowfish.h"
#include"include/version.h"
#include"include/protocol.h"

extern void fatal(const char *fmt, ...);

extern tcp_Socket s;
extern struct Packet pktin;
extern struct Packet pktout;
extern unsigned char *username;
extern char *remotehost;
extern int status;
extern DESCon keys[3];
extern BlowfishContext ectx, dctx;
extern int cipher;		/* cipher of or off*/
extern int cipher_type;	/* type of cipher */
extern char *term; /* terminal type */
extern int ssh_port; /* port number */
extern char verbose;

char privileged=1; /* Use privileged port by default */

/* check_emulation: take the remote party's version number as
   arguments and return our possibly modified version number back
   (relevant only for clients).

   Return values:
   EMULATE_VERSION_OK means we can work together

   EMULATE_VERSION_TOO_OLD if the other party has too old version
   which we cannot emulate,

   EMULATE_MAJOR_VERSION_MISMATCH if the other party has different
   major version and thus will probably not understand anything we
   say, and

   EMULATE_VERSION_NEWER if the other party has never code than we
   have.

   */

int check_emulation(int remote_major, int remote_minor,
		    int *return_major, int *return_minor)
{
  if (return_major)
    *return_major = PROTOCOL_MAJOR;
  if (return_minor)
    {
      if (remote_minor < PROTOCOL_MINOR)
	*return_minor = remote_minor;
      else
	*return_minor = PROTOCOL_MINOR;
    }

  if (remote_major < PROTOCOL_MAJOR)
    return EMULATE_MAJOR_VERSION_MISMATCH;

  if (remote_major == 1 && remote_minor == 0)
    return EMULATE_VERSION_TOO_OLD;  /* We no longer support 1.0. */
  
  if (remote_major > PROTOCOL_MAJOR ||
      (remote_major == PROTOCOL_MAJOR && remote_minor > PROTOCOL_MINOR))
    {
      /* The remote software is newer than we. If we are the client,
	 no matter - the server will decide. If we are the server, we
	 cannot emulate a newer client and decide to stop. */
      return EMULATE_VERSION_NEWER;
    }

  return EMULATE_VERSION_OK;
}


/* SSH version string exchange */
void ssh_exchange_identification(void)
{
char buf[256], remote_version[256];
int remote_major, remote_minor, i;
int my_major, my_minor;
int len, cl, dat;
unsigned int sw;

  /* Read other side's version identification. */
      i=sock_gets(&s, buf, sizeof(buf));
		if(i==0) fatal("read: %s", strerror(errno));
      buf[i] = '\n';

  if(verbose) cprintf("Remote version: %s\r",buf);
  /* Check that the versions match.  In future this might accept several
     versions and set appropriate flags to handle them. */
  if (sscanf(buf, "SSH-%d.%d-%[^\n]\n", &remote_major, &remote_minor,
	     remote_version) != 3)
    fatal("Bad remote protocol version identification\n");

  switch (check_emulation(remote_major, remote_minor,
		     &my_major, &my_minor))
    {
    case EMULATE_VERSION_TOO_OLD:
      fatal("Remote machine has too old SSH software version\n");
    case EMULATE_MAJOR_VERSION_MISMATCH:
      fatal("Major protocol versions incompatible\n");
    case EMULATE_VERSION_NEWER:
      /* We will emulate the old version. */
      break;
    case EMULATE_VERSION_OK:
      break;
    default:
      fatal("Unexpected return value from check_emulation\n");
    }

  sprintf(buf, "SSH-%d.%d-%s\n\0",
	  my_major, my_minor, SSH_VERSION);
  if(verbose) cprintf("Local version: %s\r",buf);
  len=sock_write(&s, buf, strlen(buf));
  if(len!= strlen(buf)) fatal("write: %s", strerror(errno));
}

/* create session key and ID */
void create_ssh_keys(void)
{
int i,len;
unsigned char session_key[32];
unsigned char session_id[16];
unsigned char *RSAblock, *keystr1, *keystr2;
unsigned char cookie[8];

R_RSAKey servkey, hostkey;
MD5Context md5c;

    memcpy(cookie, pktin.body, 8);
    if(verbose)cputs("Extracting server keys...");
    i=makekey(pktin.body+8, &servkey, &keystr1);
    makekey(pktin.body+8+i, &hostkey, &keystr2);
    if(verbose)cputs("Done\n\r");

    if(verbose)cputs("Generating session ID...");
    MD5Init(&md5c);
    MD5Update(&md5c, keystr2, hostkey.bytes);
    MD5Update(&md5c, keystr1, servkey.bytes);
    MD5Update(&md5c, cookie, 8);
    MD5Final(session_id, &md5c);
    if(verbose)cputs("Done\n\r");

    for (i=0; i<32; i++)
	session_key[i] = rand() % 256;

    len = (hostkey.bytes > servkey.bytes ? hostkey.bytes : servkey.bytes);

    RSAblock = malloc(len);
    if (!RSAblock)
	fatal("Out of memory\n");

    memset(RSAblock,0,len);

    for (i=0; i<32; i++) {
	RSAblock[i] = session_key[i];
	if (i < 16)
	    RSAblock[i] ^= session_id[i];
    }


    if(verbose)cputs("Encrypting session key...");
    if (hostkey.bytes > servkey.bytes) {
	rsaencrypt(RSAblock, 32, &servkey);
	rsaencrypt(RSAblock, servkey.bytes, &hostkey);
    } else {
	rsaencrypt(RSAblock, 32, &hostkey);
	rsaencrypt(RSAblock, hostkey.bytes, &servkey);
    }
    if(verbose)cputs("Done\n\r");

    s_wrpkt_start(SSH_CMSG_SESSION_KEY, len+15);
    pktout.body[0] = cipher_type;
    memcpy(pktout.body+1, cookie, 8);
    pktout.body[9] = (len*8) >> 8;
    pktout.body[10] = (len*8) & 0xFF;
    memcpy(pktout.body+11, RSAblock, len);
    pktout.body[len+11] = pktout.body[len+12] = 0;
    pktout.body[len+13] = pktout.body[len+14] = 0;
    if(verbose)cputs("Sending encrypted session key...");
    s_wrpkt();
    if(verbose)cputs("Done\n\r");
    free(RSAblock);

   switch(cipher_type)
	{
	case SSH_CIPHER_3DES:
		des_set_key(session_key, &keys[0]);
		des_set_key(session_key+8, &keys[1]);
		des_set_key(session_key+16, &keys[2]);
		break;

	case SSH_CIPHER_BLOWFISH:
		blowfish_setkey(&ectx, session_key, SSH_SESSION_KEY_LENGTH);
		    ectx.biv0 = 0;
		    ectx.biv1 = 0;
		    dctx = ectx;
		break;

	}

	cipher=1;

}

/* Send username */
void sendname(char *username)
{
int i;

    i=strlen(username);
    s_wrpkt_start(4, i+4);
    pktout.body[0] = pktout.body[1] = pktout.body[2] = 0;
    pktout.body[3] = i;
    memcpy(pktout.body+4, username, i);
    s_wrpkt();
}

/* get password */
void ask_password(void)
{
short len;
unsigned char password[30];
unsigned char ch;

	len=0;
	cputs("Password: ");
	while(ch!=0x0d && len<30)
		{
		ch=password[len]=getch();
		if(ch==8 && len>0) { password[--len]=0; continue; }
		len++;
		}
	password[len]=0;
	len--;
	cprintf("\n\r");
	s_wrpkt_start(9, len+4);
	pktout.body[0] = pktout.body[1] = pktout.body[2] = 0;
	pktout.body[3] = len;
	memcpy(pktout.body+4, password, len);
	s_wrpkt();
}


void packetsize(void)
{
int size=INBUF_SIZE;
    if(verbose)cputs("Setting maximum packet size...");
	s_wrpkt_start(SSH_CMSG_MAX_PACKET_SIZE, 4);
	pktout.body[0] = pktout.body[1] = 0;
	pktout.body[2] = size >> 8;
	pktout.body[3] = size & 0xFF;
	s_wrpkt();
	packet_read_block();
	if(verbose)
	  {
	  if(pktin.type==SSH_SMSG_SUCCESS) cputs("success");
	  else cputs("failed");
	  }
		
}


int connectssh(void)
{
int n;
int localport;
longword remoteip;

    /* allocate local port */
    localport = (rand() % 512) + 512;
    if(!privileged) localport = localport + 512;

    sock_init(); /* Initalize socket */

    if ((remoteip = resolve(remotehost)) == 0) /* Resolve hostname */
     {
       cprintf("\n\rUnable to resolve `%s'\n\r", remotehost);
       return(1);
     }

    if (!tcp_open(&s, localport, remoteip, ssh_port, NULL)) /* Open TCP */
     {
       cputs("Unable to open connection\n\r");
       return(1);
     }

/* Negotiate TCP connection */
    cputs("waiting for remote host to connect...\n\r");
    fflush(stdout);
    sock_wait_established (&s, sock_delay,NULL, &status);

    sock_wait_input (&s, sock_delay, NULL, &status);

/* Begin SSH negotiation protocol */

    /* Start negotiation on network */
    if(verbose) cputs("Identification Exchange\n\r");
    ssh_exchange_identification();
    sock_tick(&s, &status);

    /* Wait for a public key packet from the server. */
    if(verbose) cputs("Wait for public key\n\r");
    packet_read_expect(SSH_SMSG_PUBLIC_KEY);

    /* Create SSH keys */
    create_ssh_keys();

    /* Wait for key confirmation */
    if(verbose) cputs("Waiting for first encrypted ACK\n\r");
    packet_read_expect(SSH_SMSG_SUCCESS);


    /* Send username */
    sendname(username);
    packet_read_block();
    switch(pktin.type) {
	case SSH_SMSG_SUCCESS:		/* no authentication needed */
		break;
	case SSH_SMSG_FAILURE:  	/* get password */
		n=3; /* three chances */
		while(1)
		{
		 ask_password();
		 packet_read_block();
		 if(pktin.type==SSH_SMSG_SUCCESS) break;
		 n--;
		 if(n==0) fatal("Invalid password\n");
		 else cputs("Invalid password\n\r");
		}
		break;
	default:
		fatal("Invalid packet received");
	}

    /* Set maximum packet size */
    packetsize();

    return(0);

sock_err:
  switch (status)
  {
    case 1 : cputs ("Connection closed\n\r");
             break;
    case -1: cputs ("REMOTE HOST CLOSED CONNECTION\n\r");
             break;
  }
 return(1);

}
