/* protocol.c       Copyright (c) 2000 Nagy Daniel
 *
 * $Date: 2001/02/22 09:53:17 $
 * $Revision: 1.5 $
 *
 * This module deals with incoming and outgoing packets.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <mem.h>

#ifdef __DJGPP__
#include <errno.h>
#include "include/tcp_djgp.h"
#elif __TURBOC__
#include "include/tcp.h"
#endif

#include "include/ssh.h"
#include "include/des.h"
#include "include/blowfish.h"
#include "include/crc.h"

extern void fatal(const char *fmt, ...);

extern tcp_Socket s;
extern struct Packet pktin;
extern struct Packet pktout;
extern unsigned char *inbuf;
extern char debug;
extern char verbose;
extern FILE *fileptr;
extern int cipher;		/* cipher of or off*/
extern int cipher_type;         /* type of cipher */
extern int status;

DESCon keys[3];		/* 3DES contexts */
BlowfishContext ectx, dctx; /* blowfish context */

/* save packets in hexadecimal format */

static void fwritehex(unsigned char *hexbuf, unsigned int hexlength)
{
unsigned int i=0;
	for(;hexlength>0;hexlength--)
		{
		fprintf(fileptr, "%02X ", hexbuf[i]);
		i++;
		}
}

/* convert raw, encrypted packet to readable structure */
void ssh_gotdata(unsigned char *data)
{
long len, biglen;
int i, pad;

	for (i = len = 0; i < 4; i++) len = (len << 8) + data[i];
	pad = 8 - (len%8);
	biglen = len + pad;

        pktin.data = (pktin.data == NULL ? malloc(biglen) :
			realloc(pktin.data, biglen));
	    if (!pktin.data)
		fatal("Out of memory\n");

	pktin.length=biglen-4-pad;

	if(debug)
	   {
	   if(cipher) fputs("\nRECEIVED ciphertext:\n", fileptr);
	   else fputs("\nRECEIVED plaintext:\n", fileptr);
	   fwritehex(data,biglen+4);
	   fputc('\n', fileptr);
	   }

	if(cipher)
	   {
	   switch(cipher_type)
		{
		case SSH_CIPHER_3DES:
		  des_3cbc_decrypt(data+4, data+4, biglen, keys);
		break;

		case SSH_CIPHER_BLOWFISH:
		  blowfish_decrypt_cbc(data+4, biglen, &dctx);
		break;

		} /* switch */

	   if(debug)
	      {
	      fputs("\nRECEIVED plaintext:\n", fileptr);
	      fwritehex(data, biglen+4);
	      fputc('\n', fileptr);
	      } /* debug */

	  } /* cipher */

	memcpy(pktin.data, data+4, biglen);
	pktin.type = pktin.data[pad];
	pktin.body = pktin.data+pad+1;
}

/* create header for raw outgoing packet */
void s_wrpkt_start(int type, int len)
{
int pad, biglen;

    len += 5;			       /* type and CRC */
    pad = 8 - (len%8);
    biglen = len + pad;

    pktout.length = len-5;
    if (pktout.maxlen < biglen) {
	pktout.maxlen = biglen;
	pktout.data = (pktout.data == NULL ? malloc(biglen) :
		       realloc(pktout.data, biglen));
	if (!pktout.data)
	    fatal("Out of memory\n");
    }

    pktout.type = type;
    pktout.body = pktout.data+4+pad+1;
}

/* create outgoing packet */
void s_wrpkt(void)
{
int pad, len, biglen, i;
unsigned long crc;
unsigned char *p;

    len = pktout.length + 5;	       /* type and CRC */
    pad = 8 - (len%8);
    biglen = len + pad;

    pktout.body[-1] = pktout.type;
    for (i=0; i<pad; i++)
	pktout.data[i+4] = rand() % 256;
    crc = crc32(pktout.data+4, biglen-4);

    pktout.data[biglen+0] = (unsigned char) ((crc >> 24) & 0xFF);
    pktout.data[biglen+1] = (unsigned char) ((crc >> 16) & 0xFF);
    pktout.data[biglen+2] = (unsigned char) ((crc >> 8) & 0xFF);
    pktout.data[biglen+3] = (unsigned char) (crc & 0xFF);

    pktout.data[0] = (len >> 24) & 0xFF;
    pktout.data[1] = (len >> 16) & 0xFF;
    pktout.data[2] = (len >> 8) & 0xFF;
    pktout.data[3] = len & 0xFF;

    if(debug)
	{
	fputs("\nSENT plaintext:\n", fileptr);
	fwritehex(pktout.data, biglen+4);
	fputc('\n', fileptr);
	}

    if(cipher)
	{
	  switch(cipher_type)
		{
		case SSH_CIPHER_3DES:
		  des_3cbc_encrypt(pktout.data+4, pktout.data+4, biglen, keys);
		break;

		case SSH_CIPHER_BLOWFISH:
		  blowfish_encrypt_cbc(pktout.data+4, biglen, &ectx);
		break;

		} /* switch */

	  if(debug)
	    {
	    fputs("\nSENT ciphertext:\n", fileptr);
	    fwritehex(pktout.data, biglen+4);
	    fputc('\n', fileptr);
	    } /* debug */

	} /* cipher */

    if(sock_write(&s, pktout.data, biglen+4) != biglen+4)
	        fatal("write: %s", strerror(errno));

}


/* get a packet with blocking */
int packet_read_block(void)
{
short i;

 restart:

                sock_wait_input(&s,sock_delay,NULL,&status);
		sock_fastread(&s,inbuf,INBUF_SIZE);
		ssh_gotdata(inbuf);
                if(pktin.type==SSH_MSG_DEBUG)
			{
			if(debug || verbose)
			  {
			  i=pktin.body[3];
			  pktin.body[4+i]=0;
			  printf("DEBUG: %s\n",pktin.body+4);
			  }
                        goto restart;
			}
                if(pktin.type==SSH_MSG_IGNORE) goto restart;
                if(pktin.type==SSH_MSG_DISCONNECT)
                        {
			i=pktin.body[3];
			pktin.body[4+i]=0;
                        fatal("DISCONNECT: %s\n",pktin.body+4);
                        }

return(0);

sock_err:
  switch (status)
  {
    case 1 : puts ("Connection closed");
             break;
    case -1: puts ("REMOTE HOST CLOSED CONNECTION");
             break;
  }
  fatal("Socket error");
}

/* expect a packet type */
void packet_read_expect(int type)
{
short i;

      packet_read_block();
      if(pktin.type==SSH_MSG_DISCONNECT)
		{
		i=pktin.body[3];
		pktin.body[4+i]=0;
		fatal("\nDISCONNECT: %s\n",pktin.body+4);
		}
      if(pktin.type!=type) fatal("Invalid answer from server\n");
}

/* Send a character */
void ttoc(char c)
{
	s_wrpkt_start(16,5);
	pktout.body[0]=0;
	pktout.body[1]=0;
	pktout.body[2]=0;
	pktout.body[3]=1;
	pktout.body[4]=c;
	s_wrpkt();
}

/* Send a character after an ESC */
void ttoec(char c)
{
	s_wrpkt_start(16,6);
	pktout.body[0]=0;
	pktout.body[1]=0;
	pktout.body[2]=0;
	pktout.body[3]=2;
	pktout.body[4]=0x1b;
	pktout.body[5]=c;
	s_wrpkt();
}

/* Send two characters after an ESC */
void ttoe2c(char c1, char c2)
{
	s_wrpkt_start(16,7);
	pktout.body[0]=0;
	pktout.body[1]=0;
	pktout.body[2]=0;
	pktout.body[3]=3;
	pktout.body[4]=0x1b;
	pktout.body[5]=c1;
	pktout.body[6]=c2;
	s_wrpkt();
}

/* Send two characters after ANSI */
void ttoea2c(char c1, char c2)
{
	s_wrpkt_start(16,8);
	pktout.body[0]=0;
	pktout.body[1]=0;
	pktout.body[2]=0;
	pktout.body[3]=4;
	pktout.body[4]=0x1b;
	pktout.body[5]=0x5b;
	pktout.body[6]=c1;
	pktout.body[7]=c2;
	s_wrpkt();
}
/* Send three characters after ANSI */
void ttoea3c(char c1, char c2, char c3)
{
	s_wrpkt_start(16,9);
	pktout.body[0]=0;
	pktout.body[1]=0;
	pktout.body[2]=0;
	pktout.body[3]=5;
	pktout.body[4]=0x1b;
	pktout.body[5]='[';
	pktout.body[6]=c1;
	pktout.body[7]=c2;
	pktout.body[8]=c3;
	s_wrpkt();
}
