/* This is an independent implementation of the encryption algorithm:   */
/*                                                                      */
/*         RC6 by Ron Rivest and RSA Labs                               */
/*                                                                      */
/* which is a candidate algorithm in the Advanced Encryption Standard   */
/* programme of the US National Institute of Standards and Technology.  */
/*                                                                      */
/* Copyright in this implementation is held by Dr B R Gladman but I     */
/* hereby give permission for its free direct or derivative use subject */
/* to acknowledgment of its origin and compliance with any conditions   */
/* that the originators of the algorithm place on its exploitation.     */
/*                                                                      */
/* Dr Brian Gladman (gladman@seven77.demon.co.uk) 14th January 1999     */

/* $Id: rc6.c,v 1.4 2001/09/16 18:54:18 nmav Exp $ */

/* modified for mcrypt */

/* Timing data for RC6 (rc6.c)

128 bit key:
Key Setup:    1632 cycles
Encrypt:       270 cycles =    94.8 mbits/sec
Decrypt:       226 cycles =   113.3 mbits/sec
Mean:          248 cycles =   103.2 mbits/sec

192 bit key:
Key Setup:    1885 cycles
Encrypt:       267 cycles =    95.9 mbits/sec
Decrypt:       235 cycles =   108.9 mbits/sec
Mean:          251 cycles =   102.0 mbits/sec

256 bit key:
Key Setup:    1877 cycles
Encrypt:       270 cycles =    94.8 mbits/sec
Decrypt:       227 cycles =   112.8 mbits/sec
Mean:          249 cycles =   103.0 mbits/sec

*/

#include "../../lib/libdefs.h"
#include "../../lib/mcrypt_modules.h"
#include "rc6.h"

#ifdef USE_LTDL
#define _mcrypt_set_key rc6_LTX__mcrypt_set_key
#define _mcrypt_encrypt rc6_LTX__mcrypt_encrypt
#define _mcrypt_decrypt rc6_LTX__mcrypt_decrypt
#define _mcrypt_get_size rc6_LTX__mcrypt_get_size
#define _mcrypt_get_block_size rc6_LTX__mcrypt_get_block_size
#define _is_block_algorithm rc6_LTX__is_block_algorithm
#define _mcrypt_get_key_size rc6_LTX__mcrypt_get_key_size
#define _mcrypt_get_supported_key_sizes rc6_LTX__mcrypt_get_supported_key_sizes
#define _mcrypt_get_algorithms_name rc6_LTX__mcrypt_get_algorithms_name
#define _mcrypt_self_test rc6_LTX__mcrypt_self_test
#define _mcrypt_algorithm_version rc6_LTX__mcrypt_algorithm_version
#endif

#define f_rnd(i,a,b,c,d)                    \
        u = rotl32(d * (d + d + 1), 5);       \
        t = rotl32(b * (b + b + 1), 5);       \
        a = rotl32(a ^ t, u) + key->l_key[i];      \
        c = rotl32(c ^ u, t) + key->l_key[i + 1]

#define i_rnd(i,a,b,c,d)                    \
        u = rotl32(d * (d + d + 1), 5);       \
        t = rotl32(b * (b + b + 1), 5);       \
        c = rotr32(c - key->l_key[i + 1], t) ^ u;  \
        a = rotr32(a - key->l_key[i], u) ^ t

/* initialise the key schedule from the user supplied key   */

int _mcrypt_set_key(rc6_key * key, const word32 in_key[],
		     const word32 key_len)
{
	word32 i, j, k, a, b, l[8], t;

	key->l_key[0] = 0xb7e15163;

	for (k = 1; k < 44; ++k) {
		key->l_key[k] = key->l_key[k - 1] + 0x9e3779b9;
	}

	for (k = 0; k < key_len / sizeof(word32); ++k) {
#ifdef WORDS_BIGENDIAN
		l[k] = byteswap32(in_key[k]);
#else
		l[k] = in_key[k];
#endif
	}

	t = (key_len / sizeof(word32)) - 1;
/*     t = (key_len / sizeof(word32)); */

	a = b = i = j = 0;

	for (k = 0; k < 132; ++k) {
		a = rotl32(key->l_key[i] + a + b, 3);
		b += a;
		b = rotl32(l[j] + b, b);
		key->l_key[i] = a;
		l[j] = b;

		i = (i == 43 ? 0 : i + 1);
		j = (j == t ? 0 : j + 1);

/*        i = (i + 1) % 44;
 *        j = (j + 1) % t;
 */
	}
	return 0;
}

/* encrypt a block of text  */

void _mcrypt_encrypt(rc6_key * key, word32 * _blk)
{
	word32 a, b, c, d, t, u;

#ifdef WORDS_BIGENDIAN
	a = byteswap32(_blk[0]);
	b = byteswap32(_blk[1]) + key->l_key[0];
	c = byteswap32(_blk[2]);
	d = byteswap32(_blk[3]) + key->l_key[1];
#else
	a = _blk[0];
	b = _blk[1] + key->l_key[0];
	c = _blk[2];
	d = _blk[3] + key->l_key[1];
#endif

	f_rnd(2, a, b, c, d);
	f_rnd(4, b, c, d, a);
	f_rnd(6, c, d, a, b);
	f_rnd(8, d, a, b, c);
	f_rnd(10, a, b, c, d);
	f_rnd(12, b, c, d, a);
	f_rnd(14, c, d, a, b);
	f_rnd(16, d, a, b, c);
	f_rnd(18, a, b, c, d);
	f_rnd(20, b, c, d, a);
	f_rnd(22, c, d, a, b);
	f_rnd(24, d, a, b, c);
	f_rnd(26, a, b, c, d);
	f_rnd(28, b, c, d, a);
	f_rnd(30, c, d, a, b);
	f_rnd(32, d, a, b, c);
	f_rnd(34, a, b, c, d);
	f_rnd(36, b, c, d, a);
	f_rnd(38, c, d, a, b);
	f_rnd(40, d, a, b, c);

#ifndef WORDS_BIGENDIAN
	_blk[0] = a + key->l_key[42];
	_blk[1] = b;
	_blk[2] = c + key->l_key[43];
	_blk[3] = d;
#else
	_blk[0] = byteswap32(a + key->l_key[42]);
	_blk[1] = byteswap32(b);
	_blk[2] = byteswap32(c + key->l_key[43]);
	_blk[3] = byteswap32(d);
#endif

}

/* decrypt a block of text  */

void _mcrypt_decrypt(rc6_key * key, word32 * _blk)
{
	word32 a, b, c, d, t, u;

#ifndef WORDS_BIGENDIAN
	d = _blk[3];
	c = _blk[2] - key->l_key[43];
	b = _blk[1];
	a = _blk[0] - key->l_key[42];
#else
	d = byteswap32(_blk[3]);
	c = byteswap32(_blk[2]) - key->l_key[43];
	b = byteswap32(_blk[1]);
	a = byteswap32(_blk[0]) - key->l_key[42];
#endif

	i_rnd(40, d, a, b, c);
	i_rnd(38, c, d, a, b);
	i_rnd(36, b, c, d, a);
	i_rnd(34, a, b, c, d);
	i_rnd(32, d, a, b, c);
	i_rnd(30, c, d, a, b);
	i_rnd(28, b, c, d, a);
	i_rnd(26, a, b, c, d);
	i_rnd(24, d, a, b, c);
	i_rnd(22, c, d, a, b);
	i_rnd(20, b, c, d, a);
	i_rnd(18, a, b, c, d);
	i_rnd(16, d, a, b, c);
	i_rnd(14, c, d, a, b);
	i_rnd(12, b, c, d, a);
	i_rnd(10, a, b, c, d);
	i_rnd(8, d, a, b, c);
	i_rnd(6, c, d, a, b);
	i_rnd(4, b, c, d, a);
	i_rnd(2, a, b, c, d);

#ifndef WORDS_BIGENDIAN
	_blk[3] = d - key->l_key[1];
	_blk[2] = c;
	_blk[1] = b - key->l_key[0];
	_blk[0] = a;
#else
	_blk[3] = byteswap32(d - key->l_key[1]);
	_blk[2] = byteswap32(c);
	_blk[1] = byteswap32(b - key->l_key[0]);
	_blk[0] = byteswap32(a);
#endif
}



int _mcrypt_get_size()
{
	return sizeof(rc6_key);
}

int _mcrypt_get_block_size()
{
	return 16;
}

int _is_block_algorithm()
{
	return 1;
}

int _mcrypt_get_key_size()
{
	return 32;
}

char *_mcrypt_get_algorithms_name()
{
	return "RC6";
}

#define CIPHER "d0977ff486ca4f2fcfdf7bde64a8dd1c"

int _mcrypt_self_test()
{
	char *keyword;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int blocksize = _mcrypt_get_block_size(), j;
	void *key;
	unsigned char cipher_tmp[200];

	keyword = calloc(1, _mcrypt_get_key_size());
	for (j = 0; j < _mcrypt_get_key_size(); j++) {
		keyword[j] = ((j * 2 + 10) % 256);
	}
	ciphertext = malloc(blocksize);
	plaintext = malloc(blocksize);
	for (j = 0; j < blocksize; j++) {
		plaintext[j] = j % 256;
	}
	key = malloc(_mcrypt_get_size());
	memcpy(ciphertext, plaintext, blocksize);

	_mcrypt_set_key(key, (void *) keyword, _mcrypt_get_key_size());
	_mcrypt_encrypt(key, (void *) ciphertext);

	for (j = 0; j < blocksize; j++) {
		sprintf(&((char *) cipher_tmp)[2 * j], "%.2x",
			ciphertext[j]);
	}

	if (strcmp((char *) cipher_tmp, CIPHER) != 0) {
		printf("failed compatibility\n");
		printf("Expected: %s\nGot: %s\n", CIPHER,
		       (char *) cipher_tmp);
		return -1;
	}
	_mcrypt_decrypt(key, (void *) ciphertext);

	if (strcmp(ciphertext, plaintext) != 0) {
		printf("failed internally\n");
		return -1;
	}
	return 0;
}

word32 _mcrypt_algorithm_version()
{
	return 20010701;
}

static const int key_sizes[] = { 16, 24, 32 };
const int *_mcrypt_get_supported_key_sizes(int *len)
{
	*len = sizeof(key_sizes)/sizeof(int);
	return key_sizes;
}
