/* RC6 (RC6 16/20/32) Encrypt/Decrypt

    RC6 was written as a submission for AES by:

        Ronald L. Rivest, M.J.B Robshaw, R. Sidney, and Y.L Yin

    Written for Micro-C (portable) by Tom St Denis.


    The ammount of rounds and bytes per key are changeable at compile time,
but are defaulted to 20/32.  So it uses 20 rounds to encrypt 64-bits per
block using a 32 byte key (256 bits). 

Notes:

    1.  Currently if you call encrypt/decrypt seperately, it uses ECB.
    You may want to use another method such as CFB, CBC or PCBC, all of
    which can be used on top of encrypt/decrypt.

    2.  The key you pass to schedule must be at least BKEY bytes long, and
    have NULL padding if it's not that size.

    3.  The ammount of ROUNDS effects the effective strength of cipher.
    Anything between 10 and 40 is ok.  40 is a conservative high-end, and
    20 is the default.  You can also change the key length.  Keys anywhere
    from 128bits (16 bytes) to 512bits (64 bytes) are ok, but they must
    be a multiple of 2 bytes.  Also you can change the word size.  I think
    16/32 are the only really valid options, however you can use 64 too.

    4.  The memory required per scheduled key is:
            size = 4r + 8
        where r is the number of rounds.  Default is 88 bytes.

    5.  If you plan to use this with other compilers change all unsigned
    variables to unsigned short (make them 16 bits).

    6.  RC6 is a block cipher.  The default is 64-bit blocks.

    7.  Read Ronald's paper on RC6 for more detailed information.

    Micro-C is copyrighted Dave Dunfield.
    Tom St Denis, 1999
*/

#ifndef LIBDEFS_H
# define LIBDEFS_H
# include <libdefs.h>
#endif

#include <rc6.h>
#include <swap.h>

/* utility functions */
/* rotate left */
word32 rol(word32 v, word32 cnt)
{
    cnt &= (BITS - 1);

    while (cnt--)
        v = (v << 1) | (v >> (BITS - 1));

    return v;
}

/* rotate right */
word32 ror(word32 v, word32 cnt)
{
    cnt &= (BITS - 1);

    while (cnt--)
        v = (v >> 1) | (v << (BITS - 1));

    return v;
}

/* encrypt, works on 4 BITS-bit blocks at once.  The input is overwritten
with the output */
void rc6_encrypt(word32 packet[4], word32 skey[])
{
    word32 A, B, C, D, i, t, u, temp;

    /* load registers with packet */
#ifdef WORDS_BIGENDIAN
    A = byteswap(packet[0]); B = byteswap(packet[1]);
    C = byteswap(packet[2]); D = byteswap(packet[3]);
#else
    A = packet[0]; B = packet[1]; C = packet[2]; D = packet[3];
#endif

    /* start outside */
    B += skey[0]; D += skey[1];

    /* loop */
    for (i = 1; i <= RC6_ROUNDS; i++) {
        t = rol(B * ((B << 1) + 1), 2);
        u = rol(D * ((D << 1) + 1), 2);

        A = rol(A ^ t, u) + skey[i << 1];
        C = rol(C ^ u, t) + skey[(i << 1) + 1];

        /* swap */
        temp = A; A = B; B = C; C = D; D = temp;
    }

    /* finish */
    A += skey[(RC6_ROUNDS * 2) + 2];
    C += skey[(RC6_ROUNDS * 2) + 3];

    /* store packet */
#ifdef WORDS_BIGENDIAN
    packet[0] = byteswap(A); packet[1] = byteswap(B);
    packet[2] = byteswap(C); packet[3] = byteswap(D);
#else
    packet[0] = A; packet[1] = B; packet[2] = C; packet[3] = D;
#endif    
}

/* decrypt, 4 BITS-bit words.  The output overwrites the input */
void rc6_decrypt(word32 packet[4], word32 skey[])
{
    word32 A, B, C, D, i, u, t, temp;

    /* load registers with packets */
#ifdef WORDS_BIGENDIAN
    A = byteswap(packet[0]); B = byteswap(packet[1]);
    C = byteswap(packet[2]); D = byteswap(packet[3]);
#else
    A = packet[0]; B = packet[1]; C = packet[2]; D = packet[3];
#endif

    /* start */
    C -= skey[(RC6_ROUNDS * 2) + 3];
    A -= skey[(RC6_ROUNDS * 2) + 2];

    /* loop */
    for (i = RC6_ROUNDS; i >= 1; i--) {
        /* un-swap */
        temp = D; D = C; C = B; B = A; A = temp;

        u = rol(D * ((D << 1) + 1), 2);
        t = rol(B * ((B << 1) + 1), 2);
        C = ror(C - skey[(i << 1) + 1], t) ^ u;
        A = ror(A - skey[i << 1], u) ^ t;
    }

    /* finish */
    D -= skey[1];    B -= skey[0];

    /* store packet */
#ifdef WORDS_BIGENDIAN
    packet[0] = byteswap(A); packet[1] = byteswap(B);
    packet[2] = byteswap(C); packet[3] = byteswap(D);
#else
    packet[0] = A; packet[1] = B; packet[2] = C; packet[3] = D;
#endif    
    
}

/* setup the scheduled key, you must provide a buffer (the user key) with
at least BKEY bytes in it.  Fill all unused bytes with 0. */
void rc6_schedule(word8 key[], word32 skey[])
{
    word32 A, B, i, s, v, j, L[RC6_BKEY];

    /* setup local copy of key */
    for (i = 0; i < RC6_BKEY; i++)
        L[i] = (word32) key[i];

    /* standard starting values */
    skey[0] = P;

    for (i = 1; i <= ((RC6_ROUNDS * 2) + 3); i++)
        skey[i] = skey[i - 1] + Q;

    /* now setup key */
    A = B = i = j = 0;

    v = 3 * (RC6_BKEY > RC6_FACTOR ? RC6_BKEY : RC6_FACTOR);
    for (s = 1; s <= v; s++) {
        A = skey[i] = rol(skey[i] + A + B, 3);
        B = L[j] = rol(L[j] + A + B, A + B);
        i = (i + 1) % RC6_FACTOR;
        j = (j + 1) % RC6_BKEY;
    }
}





