/*
 * The GOST 28147-89 cipher
 *
 * This is based on the 25 Movember 1993 draft translation
 * by Aleksandr Malchik, with Whitfield Diffie, of the Government
 * Standard of the U.S.S.R. GOST 28149-89, "Cryptographic Transformation
 * Algorithm", effective 1 July 1990.  (Whitfield.Diffie@eng.sun.com)
 *
 * That is a draft, and may contain errors, which will be faithfully
 * reflected here, along with possible exciting new bugs.
 *
 * Some details have been cleared up by the paper "Soviet Encryption
 * Algorithm" by Josef Pieprzyk and Leonid Tombak of the University
 * of Wollongong, New South Wales.  (josef/leo@cs.adfa.oz.au)
 *
 * The standard is written by A. Zabotin (project leader), G.P. Glazkov,
 * and V.B. Isaeva.  It was accepted and introduced into use by the
 * action of the State Standards Committee of the USSR on 2 June 89 as
 * No. 1409.  It was to be reviewed in 1993, but whether anyone wishes
 * to take on this obligation from the USSR is questionable.
 *
 * This code is placed in the public domain.
 */

/* $Id: gost.c,v 1.2 1999/10/16 15:18:07 nmav Exp $ */

/*
 * If you read the standard, it belabors the point of copying corresponding
 * bits from point A to point B quite a bit.  It helps to understand that
 * the standard is uniformly little-endian, although it numbers bits from
 * 1 rather than 0, so bit n has value 2^(n-1).  The least significant bit
 * of the 32-bit words that are manipulated in the algorithm is the first,
 * lowest-numbered, in the bit string.
 */


#include "gost.h"
#ifndef SWAP_H
# define SWAP_H
# include <swap.h>
#endif

/*
 * The standard does not specify the contents of the 8 4 bit->4 bit
 * substitution boxes, saying they're a parameter of the network
 * being set up.  For illustration purposes here, I have used
 * the first rows of the 8 S-boxes from the DES.  (Note that the
 * DES S-boxes are numbered starting from 1 at the msb.  In keeping
 * with the rest of the GOST, I have used little-endian numbering.
 * Thus, k8 is S-box 1.
 *
 * Obviously, a careful look at the cryptographic properties of the cipher
 * must be undertaken before "production" substitution boxes are defined.
 *
 * The standard also does not specify a standard bit-string representation
 * for the contents of these blocks.
 */

/* These are NOT the original s-boxes. I replaced them with the ones
 * found in Applied Cryptography book by Bruce Schneier. These were
 * used in an application for the Central Bank of the Russian Federation
 * --Nikos
 */
static int init=0;

static unsigned char const k1[16] = {
	1, 15, 13, 0, 5, 7, 10, 4, 9, 2, 3, 14, 6, 11, 8, 2
};
static unsigned char const k2[16] = {
	13, 11, 4, 1, 3, 15, 5, 9, 0, 10, 14, 7, 6, 8, 2, 12
};
static unsigned char const k3[16] = {
	4, 11, 10, 0, 7, 2, 1, 13, 3, 6, 8, 5, 9, 12, 15, 14
};
static unsigned char const k4[16] = {
	6, 12, 7, 1, 5, 15, 13, 8, 4, 10, 9, 14, 0, 3, 11, 2
};
static unsigned char const k5[16] = {
	7, 13, 10, 1, 0, 8, 9, 15, 14, 4, 6, 12, 11, 2, 5, 3
};
static unsigned char const k6[16] = {
	5, 8, 1, 13, 10, 3, 4, 2, 14, 15, 12, 7, 6, 0, 9, 11
};
static unsigned char const k7[16] = {
	14, 11, 4, 12, 6, 13, 15, 10, 2, 3, 8, 1, 0, 7, 5, 9
};
static unsigned char const k8[16] = {
	4, 10, 9, 2, 13, 8, 0, 14, 6, 11, 1, 12, 7, 15, 5, 3
};

/* Byte-at-a-time substitution boxes */
static unsigned char k87[256];
static unsigned char k65[256];
static unsigned char k43[256];
static unsigned char k21[256];

/*
 * Build byte-at-a-time subtitution tables.
 * This must be called once for global setup.
 */
void _mcrypt_kboxinit(void)
{
	int i;

	if (init==0) { 
		init=1;
		for (i = 0; i < 256; i++) {
			k87[i] = k8[i >> 4] << 4 | k7[i & 15];
			k65[i] = k6[i >> 4] << 4 | k5[i & 15];
			k43[i] = k4[i >> 4] << 4 | k3[i & 15];
			k21[i] = k2[i >> 4] << 4 | k1[i & 15];
		}
	}

}

/*
 * Do the substitution and rotation that are the core of the operation,
 * like the expansion, substitution and permutation of the DES.
 * It would be possible to perform DES-like optimisations and store
 * the table entries as 32-bit words, already rotated, but the
 * efficiency gain is questionable.
 *
 * This should be inlined for maximum speed
 */
static word32 f(word32 x)
{
	/* Do substitutions */
# if 0
	/* This is annoyingly slow */
	x = k8[x >> 28 & 15] << 28 | k7[x >> 24 & 15] << 24 |
	    k6[x >> 20 & 15] << 20 | k5[x >> 16 & 15] << 16 |
	    k4[x >> 12 & 15] << 12 | k3[x >> 8 & 15] << 8 |
	    k2[x >> 4 & 15] << 4 | k1[x & 15];
# else
	/* This is faster */
	x = k87[x >> 24 & 255] << 24 | k65[x >> 16 & 255] << 16 |
	    k43[x >> 8 & 255] << 8 | k21[x & 255];
# endif

	/* Rotate left 11 bits */
	return x << 11 | x >> (32 - 11);

}

/*
 * The GOST standard defines the input in terms of bits 1..64, with
 * bit 1 being the lsb of in[0] and bit 64 being the msb of in[1].
 *
 * The keys are defined similarly, with bit 256 being the msb of key[7].
 */
void _mcrypt_gostcrypt(word32 * in, word32 const key[8])
{
	register word32 n1, n2;	/* As named in the GOST */
	word32 s_key[8];

/* Added to make it compatible with bigendian machines 
 * --nikos 
 */

#ifndef WORDS_BIGENDIAN
	n1 = byteswap(in[0]);
	n2 = byteswap(in[1]);
	s_key[0] = byteswap(key[0]);
	s_key[1] = byteswap(key[1]);
	s_key[2] = byteswap(key[2]);
	s_key[3] = byteswap(key[3]);
	s_key[4] = byteswap(key[4]);
	s_key[5] = byteswap(key[5]);
	s_key[6] = byteswap(key[6]);
	s_key[7] = byteswap(key[7]);
#else
	n1 = in[0];
	n2 = in[1];
	s_key[0] = (key[0]);
	s_key[1] = (key[1]);
	s_key[2] = (key[2]);
	s_key[3] = (key[3]);
	s_key[4] = (key[4]);
	s_key[5] = (key[5]);
	s_key[6] = (key[6]);
	s_key[7] = (key[7]);
#endif

	/* Instead of swapping halves, swap names each round */
	n2 ^= f(n1 + s_key[0]);
	n1 ^= f(n2 + s_key[1]);
	n2 ^= f(n1 + s_key[2]);
	n1 ^= f(n2 + s_key[3]);
	n2 ^= f(n1 + s_key[4]);
	n1 ^= f(n2 + s_key[5]);
	n2 ^= f(n1 + s_key[6]);
	n1 ^= f(n2 + s_key[7]);

	n2 ^= f(n1 + s_key[0]);
	n1 ^= f(n2 + s_key[1]);
	n2 ^= f(n1 + s_key[2]);
	n1 ^= f(n2 + s_key[3]);
	n2 ^= f(n1 + s_key[4]);
	n1 ^= f(n2 + s_key[5]);
	n2 ^= f(n1 + s_key[6]);
	n1 ^= f(n2 + s_key[7]);

	n2 ^= f(n1 + s_key[0]);
	n1 ^= f(n2 + s_key[1]);
	n2 ^= f(n1 + s_key[2]);
	n1 ^= f(n2 + s_key[3]);
	n2 ^= f(n1 + s_key[4]);
	n1 ^= f(n2 + s_key[5]);
	n2 ^= f(n1 + s_key[6]);
	n1 ^= f(n2 + s_key[7]);

	n2 ^= f(n1 + s_key[7]);
	n1 ^= f(n2 + s_key[6]);
	n2 ^= f(n1 + s_key[5]);
	n1 ^= f(n2 + s_key[4]);
	n2 ^= f(n1 + s_key[3]);
	n1 ^= f(n2 + s_key[2]);
	n2 ^= f(n1 + s_key[1]);
	n1 ^= f(n2 + s_key[0]);

	/* There is no swap after the last round */

#ifndef WORDS_BIGENDIAN
	in[0] = byteswap(n2);
	in[1] = byteswap(n1);
#else
	in[0] = n2;
	in[1] = n1;
#endif
}


/*
 * The key schedule is somewhat different for decryption.
 * (The key table is used once forward and three times backward.)
 * You could define an expanded key, or just write the code twice,
 * as done here.
 */
void _mcrypt_gostdecrypt(word32 * in, word32 const key[8])
{
	register word32 n1, n2;	/* As named in the GOST */
	word32 s_key[8];

#ifndef WORDS_BIGENDIAN
	n1 = byteswap(in[0]);
	n2 = byteswap(in[1]);
	s_key[0] = byteswap(key[0]);
	s_key[1] = byteswap(key[1]);
	s_key[2] = byteswap(key[2]);
	s_key[3] = byteswap(key[3]);
	s_key[4] = byteswap(key[4]);
	s_key[5] = byteswap(key[5]);
	s_key[6] = byteswap(key[6]);
	s_key[7] = byteswap(key[7]);
#else
	n1 = in[0];
	n2 = in[1];
	s_key[0] = (key[0]);
	s_key[1] = (key[1]);
	s_key[2] = (key[2]);
	s_key[3] = (key[3]);
	s_key[4] = (key[4]);
	s_key[5] = (key[5]);
	s_key[6] = (key[6]);
	s_key[7] = (key[7]);
#endif

	n2 ^= f(n1 + s_key[0]);
	n1 ^= f(n2 + s_key[1]);
	n2 ^= f(n1 + s_key[2]);
	n1 ^= f(n2 + s_key[3]);
	n2 ^= f(n1 + s_key[4]);
	n1 ^= f(n2 + s_key[5]);
	n2 ^= f(n1 + s_key[6]);
	n1 ^= f(n2 + s_key[7]);

	n2 ^= f(n1 + s_key[7]);
	n1 ^= f(n2 + s_key[6]);
	n2 ^= f(n1 + s_key[5]);
	n1 ^= f(n2 + s_key[4]);
	n2 ^= f(n1 + s_key[3]);
	n1 ^= f(n2 + s_key[2]);
	n2 ^= f(n1 + s_key[1]);
	n1 ^= f(n2 + s_key[0]);

	n2 ^= f(n1 + s_key[7]);
	n1 ^= f(n2 + s_key[6]);
	n2 ^= f(n1 + s_key[5]);
	n1 ^= f(n2 + s_key[4]);
	n2 ^= f(n1 + s_key[3]);
	n1 ^= f(n2 + s_key[2]);
	n2 ^= f(n1 + s_key[1]);
	n1 ^= f(n2 + s_key[0]);

	n2 ^= f(n1 + s_key[7]);
	n1 ^= f(n2 + s_key[6]);
	n2 ^= f(n1 + s_key[5]);
	n1 ^= f(n2 + s_key[4]);
	n2 ^= f(n1 + s_key[3]);
	n1 ^= f(n2 + s_key[2]);
	n2 ^= f(n1 + s_key[1]);
	n1 ^= f(n2 + s_key[0]);

#ifndef WORDS_BIGENDIAN
	in[0] = byteswap(n2);
	in[1] = byteswap(n1);
#else
	in[0] = n2;
	in[1] = n1;
#endif

}
