/*
 * nrm - ncrypt rm drop-in replacement
 * Attempts to be POSIX 1003.2 compliant
 * Todd MacDermid <tmacd@synacklabs.net>
 *
 * Dec 2004 - First version (that actually works)
 */

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "rand_gen.h" 
#include "wipe_file.h"
#include "nrm.h"

extern int errno;

/*
 * usage prints out the command syntax.
 */

void usage(char *programName) {
	fprintf(stderr, "usage: %s [-dfirv] file ...\n", programName);
	exit(1);
}

/*
 * check_interactive is used to ask the "do you really want to delete
 * this" kinds of questions asked when the -i flag is used. Returns
 * 1 if they really want to delete this, or 0 if they don't.
 */

int check_interactive(char *my_name, char *name, 
                      struct nrm_context *context, int nrm_type) {
  char response[10];

  if(nrm_type == NRM_DESCEND) {
    fprintf(stderr, "%s: descend into directory `%s'? ", my_name, name);
  } else if(nrm_type == NRM_DEL_DIR)  {
    fprintf(stderr, "%s: remove directory `%s'? ", my_name, name);
  } else {
    fprintf(stderr, "%s: remove `%s'? ", my_name, name);
  }
  fflush(stderr);
  fgets(response, 10, stdin);
  if((strchr(response, 'y')) || (strchr(response, 'Y'))) 
    return(1);

  return(0);    
}

/*
 * wipe_child is used to overwrite the deleted files with the requisite
 * number of passes.It makes use of the wipe_pass() function which is
 * in wipe_file.c at the time of this writing.
 */

void wipe_child(struct nrm_context *context) {
  int i;
  int j;
  int k;
  int deck[27];

  /* printf("wipe child forked with %d files\n", context->num_files); */

  seed_rand_num();

  switch (context->wipe_type) {
  case NRM_WIPE_GUTMANN:
    for(j = 0; j < 4; j++) {
      for(i = 0; i < context->num_files; i++) {
        wipe_pass(context->open_fds[i], context->blocks[i], 
		  context->extras[i], NRM_PASS_RANDOM);
      }
      for(i = 0; i < context->num_files; i++) {
        fsync(context->open_fds[i]);
      }
    }
    /* set and shuffle the deck */
    for(i=0;i<27;i++)
      deck[i] = i;
    for(i=0;i<27;i++)
      deck[i] = deck[get_rand_num(R_UNSIGNEDLONG) % 27];
    
    /* 27 passes with data to thwart forensic recovery */
    for(j=0;j<27;j++) {
      k = deck[j];
      for(i = 0; i < context->num_files; i++) {
        wipe_pass(context->open_fds[i], context->blocks[i], 
		  context->extras[i], k);
      }
      for(i = 0; i < context->num_files; i++) {
        fsync(context->open_fds[i]);
      }
    }
    /* 4 more passes with random data */
    for(j = 0; j < 4; j++) {
      for(i = 0; i < context->num_files; i++) {
        wipe_pass(context->open_fds[i], context->blocks[i], 
		  context->extras[i], NRM_PASS_RANDOM);
      }
      for(i = 0; i < context->num_files; i++) {
        fsync(context->open_fds[i]);
      }
    }
    break;
  case NRM_WIPE_SINGLE:
    for(i = 0; i < context->num_files; i++) {
      wipe_pass(context->open_fds[i], context->blocks[i], 
		context->extras[i], NRM_PASS_RANDOM);
    }
    for(i = 0; i < context->num_files; i++) {
      fsync(context->open_fds[i]);
    }
    break;
  }
  for(i = 0; i < context->num_files; i++) {
    close(context->open_fds[i]);
  }
  exit(0);
}

/*
 * reset_state closes down all the file handles opened by multiple 
 * remove_file() calls, thus allowing us to open even more file handles!
 *
 * It also resets the structure we've been using to keep track of file
 * handle state.
 */

void reset_state(struct nrm_context *context) {
  int i;
  
  for(i = 0; i < context->num_files; i++) {
    close(context->open_fds[i]);
  }
  memset(context->open_fds, 0, sizeof(int) * NUM_FD_MAX);
  memset(context->blocks, 0, sizeof(int) * NUM_FD_MAX);
  memset(context->extras, 0, sizeof(int) * NUM_FD_MAX);
  context->num_files = 0;
}

/*
 * fork_wiper forks a child off. The parent calls reset_state() to
 * close down its file handles, the child calls wipe_child() to overwrite
 * the file descriptors appropriately.
 */

int fork_wiper(struct nrm_context *context) {
  pid_t forkval;
  int keep_trying = 1;

  while(keep_trying) {
    forkval = fork();
    if(forkval > 0) {
      reset_state(context);
      keep_trying = 0;
    } else if(forkval == 0) {
      /* wipe_child does not return */
      wipe_child(context);
    } else {
      sleep(1);
    }
  }
  return(0);
}

/*
 * remove_file will open a file descriptor for the files passed in, 
 * unlink them, and return. If we've run out of file descriptors,
 * it will fork a child process off for overwriting via fork_wiper()
 *
 * Note we only overwrite files that we were the last hard link to,
 * we don't want to wax things inadvertently.
 */

int remove_file(char *my_name, char *name, struct nrm_context *context) {
  int fd;
  int i;
  int keep_trying = 1;
  struct stat statbuf;


  while(keep_trying) {
    fd = open(name, O_WRONLY);
    if(fd < 0) {
      if((EMFILE == errno) || (ENFILE == errno)) {
	if(fork_wiper(context) < 0) {
	  keep_trying = 0;
	}
      } else {

        /* Below should be vulnerable to the 'f' flag, else it should
         * prompt */

	fprintf(stderr, "%s: cannot overwrite `%s': %s\n",
		my_name, name, strerror(errno));
	return(-1);
      }
    } else {
      if(fstat(fd, &statbuf) < 0) {
	fprintf(stderr, "%s: cannot fstat `%s': %s\n",
		my_name, name, strerror(errno));
	return(-1);
      }
      if(S_ISREG(statbuf.st_mode)) {
        if(statbuf.st_nlink == 1) {
          context->open_fds[context->num_files] = fd;
          i = (int)statbuf.st_size;
          context->extras[context->num_files] = i % 256;
          context->blocks[context->num_files] = 
            (i - context->extras[context->num_files]) / 256;
          context->num_files++;
        }
      }
      if(unlink(name) < 0) {
	fprintf(stderr, "%s: cannot remove `%s': Operation not permitted\n",
		my_name, name);
	return(-1);
      }
      if(context->num_files >= NUM_FD_MAX) {
	fork_wiper(context);
      }
      return(0);
    }
  }
  /* Not reached */
  return(-1);
}

/*
 * remove_tree is really the guts of the program. It is a recursive
 * function, called on every file or directory being deleted. Note
 * that we have to treat symbolic links specially, as we want
 * to delete them, not wipe the files they point to. If we're
 * recursing, we chdir() to the subdirectory, then re-call
 * remove_tree on every directory entry. Actual real files that
 * we delete under remove_tree are passed to remove_file for
 * overwriting.
 */

int remove_tree(char *my_name, char *name, struct nrm_context *context) {
  DIR *subdir;
  struct dirent *dir_entries;
  struct stat statbuf;
  char dot[] = ".";
  char dotdot[] = "..";
  int keep_trying = 1;

  if(lstat(name, &statbuf) < 0) {
    fprintf(stderr, "%s: cannot stat `%s': %s\n", 
            my_name, name, strerror(errno));
    return(-1);
  }

  if(S_ISDIR(statbuf.st_mode)) {
    if(context->recursive_flag) {
      if(context->interactive_flag) {
        if(check_interactive(my_name, name, context, NRM_DESCEND) == 0) 
          return(0);
      }
      if(context->verbose_flag) 
        fprintf(stdout, "removing all entries of directory `%s'\n", name);
      if(chdir(name) < 0) {
        fprintf(stderr, "%s: cannot change to directory `%s': %s\n", 
                my_name, name, strerror(errno));
        return(-1);
      } 
      while(keep_trying) {
        subdir = opendir(dot);
        if(subdir == NULL) {
          if((EMFILE == errno) || (ENFILE == errno)) {
            if(fork_wiper(context) < 0) {
              fprintf(stderr, "%s: Error forking child\n", my_name);
              sleep(1);
            }
          } else {
            fprintf(stderr, "%s: cannot open dir `%s': %s\n",
                    my_name, name, strerror(errno));
            if(chdir(dotdot) < 0) {
              fprintf(stderr, 
                      "%s: Exiting - cannot get out of directory `%s': %s\n", 
                      my_name, name, strerror(errno));
              exit(-1);
            }
            return(-1);
          }
        } else {
          keep_trying = 0;
        }
      }
      while((dir_entries = readdir(subdir)) != NULL) {
	if(strcmp(dir_entries->d_name, dot) && 
	   strcmp(dir_entries->d_name, dotdot)) {
	  remove_tree(my_name, dir_entries->d_name, context);
	}
      }
      closedir(subdir);
      if(chdir(dotdot) < 0) {
        fprintf(stderr, 
                "%s: Exiting - cannot get out of directory `%s': %s\n", 
                my_name, name, strerror(errno));
        exit(-1);
      }
      if(context->interactive_flag) {
        if(check_interactive(my_name, name, context, NRM_DEL_DIR) == 0) 
          return(0);
      }
      if(context->verbose_flag) 
        fprintf(stdout, "removing the directory itself: `%s'\n", name);
      if(rmdir(name) < 0) {
        fprintf(stderr, "%s: cannot remove directory `%s': %s\n",
                my_name, name, strerror(errno));
      }
    } else {
      fprintf(stderr, "%s: `%s' is a directory\n", my_name, name);
    }
  } else if((S_IFLNK & statbuf.st_mode) == S_IFLNK) { 
    if(context->interactive_flag) {
      if(check_interactive(my_name, name, context, NRM_DEL_FILE) == 0) 
        return(0);
    }
    if(context->verbose_flag) 
      fprintf(stdout, "removing `%s'\n", name);
    if(unlink(name) < 0) {
      fprintf(stderr, "%s: cannot remove `%s': %s\n",
              my_name, name, strerror(errno));
      return(-1);
    }
  } else if(S_ISREG(statbuf.st_mode)) {
    if(context->interactive_flag) {
      if(check_interactive(my_name, name, context, NRM_DEL_FILE) == 0) 
        return(0);
    }
    if(context->verbose_flag) 
      fprintf(stdout, "removing `%s'\n", name);
    remove_file(my_name, name, context);
  } else {
    if(context->interactive_flag) {
      if(check_interactive(my_name, name, context, NRM_DEL_FILE) == 0) 
        return(0);
    }
    if(context->verbose_flag) 
      fprintf(stdout, "removing `%s'\n", name);
    if(unlink(name) < 0) {
      fprintf(stderr, "%s: cannot remove `%s': %s\n",
              my_name, name, strerror(errno));
      return(-1);
    }
  }

  return(0);
}

/*
 * The main loop. This simply parses options, sets flags, and then calls
 * remove_tree on each non-flag argument. At the end forks off a child
 * to wipe any file handles left over.
 */ 

int main(int argc, char **argv) {
  int c, i;
  struct nrm_context context;

  memset(&context, 0, sizeof(context));
  context.wipe_type = NRM_WIPE_GUTMANN;

  while((c=getopt(argc, argv, "dfirRv")) != -1) {
    switch(c) {
    case 'd':
      context.dir_flag = 1;
      break;
    case 'f':
      context.force_flag = 1;
      break;
    case 'i':
      context.interactive_flag = 1;
      break;
    case 'r':
    case 'R':
      context.recursive_flag = 1;
      break;
    case 'v':
      context.verbose_flag = 1;
      break;
    default:
      usage(argv[0]);
    }
  } 
   
  if((optind >= argc) && (!context.force_flag)) 
      usage(argv[0]);

  for(i = optind; i < argc; i++) {
    remove_tree(argv[0], argv[i], &context);
  }
  fork_wiper(&context);
  return(0);
}
