#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>

extern char *strdup(const char *);

#include "netlib3.h"
#include "netperf3.h"
#include "nettest3_bsd.h"

const char sockets_usage[] = "\n\
Usage: netperf [global options] -- [test options] \n\
\n\
TCP/UDP BSD Sockets Test Options:\n\
    -D [L][,R]        Set TCP_NODELAY locally and/or remotely (TCP_*)\n\
    -h                Display this text\n\
    -H name|ip        Override the global -H as the test destination\n\
    -L name|ip        Explicitly set the source IP address\n\
    -m bytes          Set the send size (TCP_STREAM, UDP_STREAM)\n\
    -M bytes          Set the recv size (TCP_STREAM, UDP_STREAM)\n\
    -p min[,max]      Set the min/max port numbers for TCP_CRR, TCP_TRR\n\
    -P src[,dst]      Explicitly set the src and dst port numbers\n\
    -r req[,rsp]      Set request and response size (TCP_RR, UDP_RR)\n\
    -s send[,recv]    Set local socket send/recv buffer sizes\n\
    -S send[,recv]    Set remote socket send/recv buffer sizes\n\
    -w bytes[,bytes]  Set the local/remote socket watermarks\n\
\n\
For those options taking two parms, at least one must be specified;\n\
specifying one value without a comma will set both parms to that\n\
value, specifying a value with a leading comma will set just the second\n\
parm, a value with a trailing comma will set just the first. To set\n\
each parm to unique values, specify both and separate them with a\n\
comma.\n"; 

void
print_sockets_usage()
{

  printf("%s",sockets_usage);
  exit(1);

}

void
retrieve_socket_settings(int temp_socket, test_t *test)
{
  bsd_test_results *results;
  bsd_test_settings *final_settings;

  int sock_opt_len;

  results = (bsd_test_results *)&(test->test_specific_results);
  final_settings = (bsd_test_settings *)&(results->final_settings);

  if (debug) {
    fprintf(where,
	    "retrieve_socket_settings called socket %d test %p\n",
	    temp_socket,test);
    fflush(where);
  }

#ifdef TCP_MAXSEG  
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 /* hmm, is this really such a good idea? */
		 getprotobyname("tcp")->p_proto,
		 TCP_MAXSEG,
		 (char *)&(results->segment_size),
		 &sock_opt_len) != 0) {
    fprintf(where,
	    "Thread %d could not retrieve the TCP_MSS - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->segment_size = -1;
  }
#else /* TCP_MAXSEG */
  /* The TCP_MAXSEG option is not available */
  results->segment_size = -2;
#endif /* TCP_MAXSEG */

#ifdef SO_SNDBUF
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_SNDBUF,
		 (char *)&(results->final_settings.local_socket_send),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "Thread %d could not retrieve final SO_SNDBUF - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->final_settings.local_socket_send = -1;
  }
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_RCVBUF,
		 (char *)&(results->final_settings.local_socket_recv),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "Thread %d could not retreive final SO_RCVBUF - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->final_settings.local_socket_recv;
  }
  
#else /* SO_SNDBUF */
  
  results->final_settings.local_socket_send = -1;
  results->final_settings.local_socket_recv = -1;
  
#endif /* SO_SNDBUF */

}

void
dump_bsd_settings_i(bsd_test_settings *settings)
{
  if (settings == NULL) {
    fprintf(where,"Cannot dump settings from a null pointer\n");
    exit(-1);
  }

  fprintf(where,"\ttest_dest |%s|\n",settings->test_dest);
  fprintf(where,"\ttest_source |%s|\n",settings->test_source);
  fprintf(where,"\tlocal_socket_send %d\n",settings->local_socket_send);
  fprintf(where,"\tlocal_socket_recv %d\n",settings->local_socket_recv);
  fprintf(where,"\tlocal_nodelay %d\n",settings->local_nodelay);
  fprintf(where,"\tsend_size %d\n",settings->send_size);
  fprintf(where,"\trequest_size %d\n",settings->request_size);
  fprintf(where,"\tlocal_watermark %d\n",settings->local_watermark);
  fprintf(where,"\tlocal_port_min %d\n",settings->local_port_min);
  fprintf(where,"\tlocal_port_max %d\n",settings->local_port_max);
  fprintf(where,"\tlocal_port_num %d\n",settings->local_port_num);
  fprintf(where,"\tremote_port_num %d\n",settings->remote_port_num);
  fprintf(where,"\tremote_socket_send %d\n",settings->remote_socket_send);
  fprintf(where,"\tremote_socket_recv %d\n",settings->remote_socket_recv);
  fprintf(where,"\tremote_nodelay %d\n",settings->remote_nodelay);
  fprintf(where,"\trecv_size %d\n",settings->recv_size);
  fprintf(where,"\tresponse_size %d\n",settings->response_size);
  fprintf(where,"\tremote_watermark %d\n",settings->remote_watermark);
}

void
dump_bsd_settings(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents for bsd_test_settings for thread %d\n",
	  test->thread_num);
  dump_bsd_settings_i((bsd_test_settings *)&(test->test_specific_settings));
}

static void
dump_bsd_results_i(bsd_test_results *results)
{
  if (results == NULL) {
    fprintf(where,"Cannot dump results from a null pointer\n");
    exit(-1);
  }
  fprintf(where,
	  "Final Settings\n");
  dump_bsd_settings_i((bsd_test_settings *)&(results->final_settings));

  fprintf(where,
	  "Final Results\n");
  fprintf(where,"\tbytes_received %d\n",results->bytes_received);
  fprintf(where,"\tnum_sends %d\n",results->num_sends);
  fprintf(where,"\tnum_receives %d\n",results->num_receives);
  fprintf(where,"\tsegment_size %d\n",results->segment_size);
}

void
dump_bsd_results(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents of bsd_test_results for thread %d\n",
	  test->thread_num);
  dump_bsd_results_i((bsd_test_results *)test->test_specific_results);

}

 /* This routine will create a data (listen) socket with the apropriate */
 /* options set and return it to the caller. this replaces all the */
 /* duplicate code in each of the test routines and should help make */
 /* things a little easier to understand. since this routine can be */
 /* called by either the netperf or netserver programs, all output */
 /* should be directed towards "where." family is generally AF_INET, */
 /* and type will be either SOCK_STREAM or SOCK_DGRAM */
static int
create_data_socket(int family, int type, test_t *test)
{

  int temp_socket;
  int one;
  int sock_opt_len;
  uint32_t addr;
  struct sockaddr_in myaddr;
  struct hostent *hp;
  int32_t do_bind = 0;

  /* from this we will get what the values should be */
  bsd_test_settings *settings;

  /* and here we may put what the values became */
  bsd_test_results  *results;

  /* we need some VASSERTS here I think */
  settings = (bsd_test_settings *)&(test->test_specific_settings);
  results  = (bsd_test_results *)&(test->test_specific_results);

  /* set up the data socket */
  temp_socket = socket(family, type,  0);
  
#ifdef WIN32
  if (temp_socket == INVALID_SOCKET){
#else
  if (temp_socket < 0){
#endif /* WIN32 */
    fprintf(where,
	    "netperf: create_data_socket: socket: %d\n",
	    errno);
    fflush(where);
    exit(-1);
  }
  
  if (debug) {
    fprintf(where,"create_data_socket: socket %d obtained...\n",temp_socket);
    fflush(where);
  }
  
  /* Modify the local socket size. The reason we alter the send buffer */
  /* size here rather than when the connection is made is to take care */
  /* of decreases in buffer size. Decreasing the window size after */
  /* connection establishment is a TCP no-no. Also, by setting the */
  /* buffer (window) size before the connection is established, we can */
  /* control the TCP MSS (segment size). The MSS is never more that 1/2 */
  /* the minimum receive buffer size at each half of the connection. */
  /* This is why we are altering the receive buffer size on the sending */
  /* size of a unidirectional transfer. If the user has not requested */
  /* that the socket buffers be altered, we will try to find-out what */
  /* their values are. If we cannot touch the socket buffer in any way, */
  /* we will set the values to -1 to indicate that.  */
  
#ifdef SO_SNDBUF
  if (settings->local_socket_send > 0) {
    if(setsockopt(temp_socket, SOL_SOCKET, SO_SNDBUF,
		  (char *)&(settings->local_socket_send), 
		  sizeof(int)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: SO_SNDBUF option: errno %d\n",
	      errno);
      fflush(where);
      exit(1);
    }
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: SO_SNDBUF of %d requested.\n",
	      settings->local_socket_send);
      fflush(where);
    }
  }
  if (settings->local_socket_recv > 0) {
    if(setsockopt(temp_socket, SOL_SOCKET, SO_RCVBUF,
		  (char *)&(settings->local_socket_recv),
		  sizeof(int)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: SO_RCVBUF option: errno %d\n",
	      errno);
      fflush(where);
      exit(1);
    }
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: SO_RCVBUF of %d requested.\n",
	      settings->local_socket_recv);
      fflush(where);
    }
  }
  
  
  /* Now, we will find-out what the size actually became, and report */
  /* that back to the user. If the call fails, we will just report a -1 */
  /* back to the initiator for the recv buffer size. */
  
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_SNDBUF,
		 (char *)&(results->final_settings.local_socket_send),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "netperf: create_data_socket: getsockopt SO_SNDBUF: errno %d\n",
	    errno);
    fflush(where);
    results->final_settings.local_socket_send = -1;
  }
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_RCVBUF,
		 (char *)&(results->final_settings.local_socket_recv),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "netperf: create_data_socket: getsockopt SO_SNDBUF: errno %d\n",
	    errno);
    fflush(where);
    results->final_settings.local_socket_recv;
  }
  
  if (debug) {
    fprintf(where,
	    "netperf: create_data_socket: socket sizes determined...\n");
    fprintf(where,
	    "                       send: %d recv: %d\n",
	    results->final_settings.local_socket_send,
	    results->final_settings.local_socket_recv);
    fflush(where);
  }
  
#else /* SO_SNDBUF */
  
  results->final_settings.local_socket_send = -1;
  results->final_settings.local_socket_recv = -1;
  
#endif /* SO_SNDBUF */

  
  /* Now, we will see about setting the TCP_NODELAY flag on the local */
  /* socket. We will only do this for those systems that actually */
  /* support the option. If it fails, note the fact, but keep going. */
  /* If the user tries to enable TCP_NODELAY on a UDP socket, this */
  /* will cause an error to be displayed */
  
#ifdef TCP_NODELAY
  if (settings->local_nodelay) {
    one = 1;
    if(setsockopt(temp_socket,
		  getprotobyname("tcp")->p_proto,
		  TCP_NODELAY,
		  (char *)&one,
		  sizeof(one)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: nodelay: errno %d\n",
	      errno);
      fflush(where);
    }
    
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: TCP_NODELAY requested...\n");
      fflush(where);
    }
  }
#else /* TCP_NODELAY */
  
  settings->local_nodelay = 0;
  
#endif /* TCP_NODELAY */

  bzero(&myaddr,sizeof(myaddr));
  myaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  myaddr.sin_port = htons(0);
  myaddr.sin_family = AF_INET;
  
  /* if the user specified a different source IP address or host name,
     look that up.  */  

  if (strcmp(settings->test_source,"INADDR_ANY") != 0) {
    /* looks as though someone wanted to specify a local address */
    if (debug) {
      fprintf(where,"Asked to set the local address to %s\n",
	      settings->test_source);
    }

    if ((addr = inet_addr(settings->test_source)) == -1) {
      do_bind++;
      /* it was not an IP address, try it as a name */
      if ((hp = gethostbyname(settings->test_source)) == NULL) {
	/* we have no idea what address they wwere asking for */
	fprintf(where,
		"Could not translate %s for binding to a socket\n",
		settings->test_source);
	exit(-1);
      }
      else {
	/* it was a valid remote_host */
	bcopy(hp->h_addr,
	      (char *)&myaddr.sin_addr,
	      hp->h_length);
	myaddr.sin_family = hp->h_addrtype;
      }
    }
    else {
      /* it was a valid IP address */
      myaddr.sin_addr.s_addr = addr;
      myaddr.sin_family = AF_INET;
    }
  }

  /* now, we see if the user wanted to bind the socket to a specific
     port number or IP address */
  if (settings->local_port_num) {
    myaddr.sin_port = htons(settings->local_port_num);
    do_bind++;
  }

  /* ok, now myaddr has all the pertinent information, call bind */
  if (do_bind) {
    printf("asked to bind\n");
    if (bind(temp_socket,
	     (struct sockaddr *)&myaddr,
	     sizeof(struct sockaddr_in)) < 0) {
      /* well, that didn't work so well, did it? */
      fprintf(where,
	      "create_data_socket could not bind errno %d addr %s port %d\n",
	      errno,
	      inet_ntoa(myaddr.sin_addr),
	      ntohs(myaddr.sin_port));
      exit(-1);
    }
  }
  return(temp_socket);

}

void
init_bsd_settings(bsd_test_settings *settings)
  {
    /* this will get fleshed-out more later */
    strcpy(settings->test_source,"INADDR_ANY");
    strcpy(settings->test_dest,"INADDR_ANY");
  }

void
scan_bsd_args(test_t *test)
{

  /* it is important to remember that getopt is *NOT* thread-safe, so
     we want to make very certain that this routine is _only_ called
     from the main thread. */

  extern char	*optarg;	  /* pointer to option string	*/
  
  int		c;

  bsd_test_settings *settings;

  char	
    arg1[BUFSIZ],  /* argument holders		*/
    arg2[BUFSIZ];
  
  /* make sure we were not given a NULL pointer to our test structure */
  assert((test != NULL));

  /* make sure we have enough space for our test-specific settings. I
     would have prefered this to be compile-time check, but one cannot
     put a sizeof() into a #if. bummer. raj 2/98 */
  assert((sizeof(bsd_test_settings) < NETPERF_MAX_SETTINGS));

  settings = (bsd_test_settings *)test->test_specific_settings;

  /* while we are here, we might as well check the results structure
     too */

  init_bsd_settings(settings);

  assert((sizeof(bsd_test_results) < NETPERF_MAX_RESULTS));


  /* Go through all the command line arguments and break them */
  /* out. For those options that take two parms, specifying only */
  /* the first will set both to that value. Specifying only the */
  /* second will leave the first untouched. To change only the */
  /* first, use the form "first," (see the routine break_args.. */
  
  while ((c= getopt(test->argc, test->argv, SOCKETS_ARGS)) != EOF) {
    switch (c) {
    case '?':	
    case 'h':
      print_sockets_usage();
      exit(1);
    case 'D':
      /* set the TCP nodelay flag */
      settings->local_nodelay = 1;
      settings->remote_nodelay = 1;
      break;
    case 'H':
      /* the user wishes to specify a remote hostname or IP address
	 which differs from the one used to establish the control
	 connection. of course if this path can stand the test
	 connection, it should be able to stand the control connection
	 too */
      strncpy(settings->test_dest,optarg,NETPERF_HOST_MAX);
      break;
    case 'L':
      /* the user wishes to specify the source IP address (perhaps
	 using a hostname) from which the test should connect. I am
	 not quite sure at the moment how the "sense" of these should
	 be for a TCP_STREAM versus a TCP_MAERTS test for example. */
      strncpy(settings->test_source,optarg,NETPERF_HOST_MAX);
      break;
    case 's':
      /* set local socket sizes */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->local_socket_send = convert(arg1);
      if (arg2[0])
	settings->local_socket_recv = convert(arg2);
      break;
    case 'S':
      /* set remote socket sizes */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->remote_socket_send = convert(arg1);
      if (arg2[0])
	settings->remote_socket_recv = convert(arg2);
      break;
    case 'r':
      /* set the request/response sizes */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->request_size = convert(arg1);
      if (arg2[0])	
	settings->response_size = convert(arg2);
      break;
    case 'm':
      /* set the send size */
      settings->send_size = convert(optarg);
      break;
    case 'M':
      /* set the recv size */
      settings->recv_size = convert(optarg);
      break;
    case 'p':
      /* set the min and max port numbers for the TCP_CRR and TCP_TRR */
      /* tests. */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->local_port_min = (uint16_t) atoi(arg1);
      if (arg2[0])	
	settings->local_port_max = (uint16_t) atoi(arg2);
      break;
    case 'P':
      /* set the source and destination port numbers for the basic
	 send and receive tests */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->local_port_num = (uint16_t) atoi(arg1);
      if (arg2[0])	
	settings->remote_port_num = (uint16_t) atoi(arg2);
      break;
    case 'w':
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->local_watermark = (uint16_t) atoi(arg1);
      if (arg2[0])	
	settings->remote_watermark = (uint16_t) atoi(arg2);
      break;
    };
  }
}

void
send_tcp_stream(test_t *test)
{

  int data_socket;
  char *test_dest;
  int test_num;
  char test_message[MAX_CONTROL_MSG_LEN];
  int32_t bytes_to_send;
  int32_t bytes_sent;

  struct sockaddr_in server;
  struct hostent *hp;
  int32_t addr;

  ring_elt_t *send_ring = NULL;

  bsd_test_settings *settings;
  bsd_test_settings *final_settings;
  bsd_test_results  *results;

  settings = (bsd_test_settings *)&(test->test_specific_settings);

  results = (bsd_test_results *)&(test->test_specific_results);

  final_settings = (bsd_test_settings *)&(results->final_settings);

  if (debug) {
    fprintf(where,
	    "Sending a TCP_STREAM test settings %p results %p final %p\n",
	    settings,
	    results,
	    final_settings);
    fflush(where);
  }

  if (debug > 1) {
    dump_bsd_settings(test);
  }

  /* after this call, we will have a socket with all our settings,
     locally bound to what may have been specified by the user */
  data_socket = create_data_socket(AF_INET,
				   SOCK_STREAM,
				   test);


  /* at this point, we have either retrieved the socket buffer sizes,
     or have tried to set them, so now it is time to select a
     send_size if it was not preselected by the user. if we have no
     idea what the socket buffer size is (say we are running on a
     bogus implementation) we will make the send size an arbitrary
     4096 bytes. raj 2/98 */

  if (settings->send_size == 0) {
    if (final_settings->local_socket_send > 0) {
      final_settings->send_size = final_settings->local_socket_send;
    }
    else {
      final_settings->send_size = 4096;
    }
  }
  else {
    final_settings->send_size = settings->send_size;
  }

  /* now that we know the send size, we can allocate our
     send_ring. the user may not have specified a number of send
     buffers, in which case we will pick a number based on the
     send_size and the send socket buffer size, with a minimum of two
     buffers. raj 2/98 */

  if (test->send_width == 0) {
    test->send_width = 1 +
      (final_settings->local_socket_send / final_settings->send_size);
    if (test->send_width == 1) test->send_width++;
  }

  /* with the send_size, and the send_width determined, we can now go
     ahead an allocate our buffer ring. I'm not really sure that I
     want the buffers allocated from the shared memory in a threaded
     test but I also don't want to worry about support for alloca or
     anything else like that. should someone find that simply using
     malloc() is leading to inapropriate results, we could see about
     using alloca() to allocate the buffers from the heap, in which
     case, there should be no relationship among the buffers in a
     threaded test. raj 2/98 */

  if (send_ring == NULL) {
    /* give me a ring, one ring only please vasilli... */
    send_ring = allocate_buffer_ring(test->send_width,
				     final_settings->send_size,
				     test->local_send_align,
				     test->local_send_offset,
				     test->fill_file);
  }

  /* if the user requested CPU utilization measurements, we want to
     get the CPU measurement infrastructure started and warmed-up. */

  calibrate_local_cpu(test);

  /* now lets lookup the information for the test destination since we
   are now disconnected from the code that established the control
   socket, and since we want to be able to use different protocols and
   such, we are passed the name of the remote host and must turn that
   into the test specific addressing information. */
  
  bzero((char *)&server,
	sizeof(server));
  
  /* it would seem that while HP-UX will allow an IP address (as a
     string) in a call to gethostbyname, other, less enlightened
     systems do not. fix from awjacks@ca.sandia.gov raj 10/95 order
     changed to check for IP address first. raj 7/96 */ 

  if (strcmp(settings->test_dest,"INADDR_ANY") == 0) {
    /* the user did not override the destination */
    test_dest = test->remote_host;
  }
  else {
    /* the user did override the destination */
    test_dest = settings->test_dest;
  }

  if ((addr = inet_addr(test_dest)) == -1) {
    /* it was not an IP address, try it as a name */
    if ((hp = gethostbyname(test_dest)) == NULL) {
      /* we have no idea what it is */
      fprintf(where,
	      "send_tcp_stream: could not resolve the destination %s\n",
	      test_dest);
      fflush(where);
      exit(1);
    }
    else {
      /* it was a valid remote_host */
      bcopy(hp->h_addr,
	    (char *)&server.sin_addr,
	    hp->h_length);
      server.sin_family = hp->h_addrtype;
    }
  }
  else {
    /* it was a valid IP address */
    server.sin_addr.s_addr = addr;
    server.sin_family = AF_INET;
  }    
  

  /* now that we are ready on our side, lets tell the remote what we
     want to do. admittedly, this is a triffle ugly - i guess that if
     I wanted to get fancy, I could define a parser for the format
     string that would automagically pick the proper variables, but
     that is a luxury. REMEMBER, we are sending the *remote* the
     settings it needs to know. raj 2/98 */

  sprintf(test_message,
	  TCP_STREAM_REQ,
	  /* first, the NETPERF_BUF_SETTINGS */
	  test->remote_cpu_rate,
	  test->remote_send_align,
	  test->remote_recv_align,
	  test->remote_send_offset,
	  test->remote_recv_offset,
	  test->send_width,
	  test->recv_width,
	  /* now, the TCP_STREAM_REQ_COMM settings */
	  settings->remote_socket_send,
	  settings->remote_socket_recv,
	  settings->recv_size,
	  settings->remote_port_num,
	  settings->remote_nodelay,
	  settings->remote_watermark,
	  settings->test_dest);

  send_control_message(test->control_sock,
		       TCP_STREAM,
		       strlen(test_message),
		       test_message);

  
  /* once the remote has chewed on our request, he will send us a
     response */

  test_num = recv_control_message(test->control_sock,
				  0, /* we want the select */
				  sizeof(test_message),
				  test_message);

  if (test_num != TCP_STREAM) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",test_message);
    fflush(where);
    exit(-1);
  }
	
  /* so, at this point, we know that everything  is hunky-dory, so
     lets grab the information from the remote */

  sscanf(test_message,
	 TCP_STREAM_RSP,
	 &(final_settings->remote_socket_send),
	 &(final_settings->remote_socket_recv),
	 &(final_settings->recv_size),
	 &(final_settings->remote_port_num),
	 &(final_settings->remote_nodelay),
	 &(final_settings->remote_watermark),
	 /* remember, test_dest is a string ! */
	 final_settings->test_dest);

  /* use the port number given to us by the remote - be sure the port
     number is in network byte order */
  server.sin_port = htons(final_settings->remote_port_num);

  /* we now wait for everyone else to say that they are ready */
  barrier_wait(netperf_barrier);

  /* the test starts here. it seems reasonable to include the connect()
   time in the measurement, one could go the other way on the premis
   that this is just supposed to measure the transfer rate, but that
   same reasoning would suggest that it is ok to wait until after a
   window's worth of data has been transfered to start timing  so
   slow-start is avoided. better overall to include the connect
   time. raj 2/98 */

  /* ok, now set-up the conditions to tell us when it is time to stop
     sending data. in prior versions of netperf, this could be either
     time, or byte-count based, however, for netperf3, this will just
     be time-based until someone squawks. looks like I might be able
     to combine a couple calls here, but I'll wait until no-one
     squawks about the dropping of byte-length tests. raj 2/98 */
  
  start_timer(test);
  cpu_start(test);

  /* Connect up to the remote port on the data socket  */
  if (connect(data_socket, 
	      (struct sockaddr *)&server,
	      sizeof(server)) < 0 ) {
    fprintf(where,
	    "Thread %d's data connection to %s port %d failed - errno %d",
	    test->thread_num,
	    test_dest,
	    final_settings->remote_port_num,
	    errno);
    fflush(where);
    exit(-1);
  }

  while (!test->times_up) {
    bytes_to_send = final_settings->send_size;
    while (bytes_to_send > 0) {
      bytes_sent = send(data_socket,
			send_ring->buffer_ptr,
			final_settings->send_size,
			0);
      /* seems that on some systems (Digital Unix 4.0 and HP-UX 11.00
	 I cannot rely on SA_RESTART always restarting a send system
	 call. so, I have to have a check for EINTR, and I probably
	 need to do aloop to make sure I send "send_size" bytes each
	 time - this is a little strange, but the slop should only hit
	 at the very end of the test, so it should not be a big deal
	 that the last "send" was really two sometimes. this is really
	 only needed when I am using processes and signals to end the
	 test - the threaded version should never hit this at all. raj
	 2/98 */ 
      if ((bytes_sent < 0) && (errno != EINTR)) {
	fprintf(stderr,
		"Thread %d encountered a send() failure - errno %d\n",
		test->thread_num,
		errno);
	exit(-1);
      }
      else {
	bytes_to_send -= bytes_sent;
      }
    }
    /* ok, that send did not fail, up the number of send calls and
       point ourselves at the next buffer in the ring */

    results->num_sends++;
    send_ring = send_ring->next;
  }

  retrieve_socket_settings(data_socket,test);


  /* ok, if we got here, that means we've exited from the while
     loop. we will execute a graceful (aka PROPER!) shutdown of the
     data connection to make sure that all data was received by the
     remote. none of this silly abortive close stuff... raj 2/98 */

  /* well, there are some systems which do not set the socket buffer
     information until *after* the connection is established. while I
     personally feel that such systems are a triffle silly, I hve to
     be at least a *little* fair and make a call here to retrieve the
     socket values. I'll sugar coat this for myself by retrieving the
     TCP MSS for the test at the same time. If nothing else, this
     should be happening in parallel with the last bytes of data
     getting to the remote, so we won't affect throughput very much,
     we might consume a few extra cycles of CPU time, but not enough
     to be worth worrying about. I suppose that it could be moved
     outside the timing window, but then there could be a slight
     discrepancy with the behaviour of something like a TCP_CRR or
     TCP_CC test? */


  /* ok, _now_ can I have me some fighting room!?! :) */
  if (shutdown(data_socket, 1) != 0) {
    /* um, houston, I don't think the data is in Kansas anymore */
    fprintf(stderr,
	    "Thread %d could not flush its data to the remote - errno %d\n",
	    test->thread_num,
	    errno);
    exit(-1);
  }

  /* according to our little "protocol" here, the remote will also do
     a shutdown, which according to generally accepted socket
     semantics, should get us out or our recv() call. if the return
     value is other than zero, punt. raj 2/98 */
  if (recv(data_socket,
	   send_ring->buffer_ptr, 
	   final_settings->send_size,
	   0) != 0) {
    fprintf(stderr,
	    "Thread %d did not receive zero bytes at test end - errno %d\n",
	    test->thread_num,
	    errno);
    exit(-1);
  }

  cpu_stop(test);

  /* this test iteration is over, now sync-up with everyone else -
     either for output, or for the next iteration. */
  barrier_wait(netperf_stop_barrier);
}

void
recv_tcp_stream(test_t *test)
{
  bsd_test_settings *settings;
  bsd_test_settings *final_settings;
  bsd_test_results *results;

  int listen_socket;
  int data_socket;

  int len;
  
  int addrlen;
  struct sockaddr_in myaddr_in;
  struct sockaddr_in peeraddr_in;

  ring_elt_t *recv_ring;

  char reply_message[MAX_CONTROL_MSG_LEN];

  settings = (bsd_test_settings *)&(test->test_specific_settings);

  results = (bsd_test_results *)&(test->test_specific_results);

  final_settings = (bsd_test_settings *)&(results->final_settings);

  printf("Receiving a TCP_STREAM test\nMy initial message was |%s|\n",
	 test->setup_message);

  /* pull the information from the test message */
  sscanf(test->setup_message,
	 TCP_STREAM_REQ,
	 /* first, the NETPERF_BUF_SETTINGS */
	 &(test->local_cpu_rate),
	 &(test->local_send_align),
	 &(test->local_recv_align),
	 &(test->local_send_offset),
	 &(test->local_recv_offset),
	 &(test->send_width),
	 &(test->recv_width),
	  /* now, the TCP_STREAM_REQ_COMM settings */
	 &(settings->local_socket_send),
	 &(settings->local_socket_recv),
	 &(settings->recv_size),
	 &(settings->local_port_num),
	 &(settings->local_nodelay),
	 &(settings->local_watermark),
	 /* test source is a string!!! */
	 &(settings->test_source[0])); 

  dump_bsd_settings(test);

  /* having placed the relevant settings into settings, now we can
     create the data socket, any requested binding to specific
     addresses will happen in this routine */
  listen_socket = create_data_socket(AF_INET,
				     SOCK_STREAM,
				     test);

  /* now we can hang a listen on this socket */
  if (listen(listen_socket, 5) == -1) {
    sprintf(reply_message,
	    "recv_tcp_stream: listen failled errno %d",
	    errno);
    send_control_message(test->control_sock,
			 ERROR,
			 strlen(reply_message),
			 reply_message);
    close(listen_socket);
    /* is exit really what I want to do? */
    exit(-1);
  }

  /* it would be a good idea to pick an apropriate recv size */
  if (settings->recv_size == 0) {
    if (final_settings->local_socket_recv > 0) {
      final_settings->recv_size = final_settings->local_socket_recv;
    }
    else {
      final_settings->recv_size = 4096;
    }
  }
  else {
    final_settings->recv_size = settings->recv_size;
  }

  if (test->recv_width == 0) {
    test->recv_width = 
      (final_settings->local_socket_recv/final_settings->recv_size) + 1;
    if (test->recv_width == 1) test->recv_width++;
  }

  recv_ring = allocate_buffer_ring(test->recv_width,
				   final_settings->recv_size,
				   test->local_recv_align,
				   test->local_recv_offset,
				   test->fill_file);

  if (debug) {
    fprintf(where,"recv_tcp_stream: receive alignment and offset set...\n");
    fflush(where);
  }

  /* now get the port number assigned by the system  */
  addrlen = sizeof(myaddr_in);
  if (getsockname(listen_socket, 
		  (struct sockaddr *)&myaddr_in,
		  &addrlen) == -1){
    sprintf(reply_message,
	    "recv_tcp_stream: getsockname failed errno %d\n",
	    errno);
    send_control_message(test->control_sock,
			 ERROR,
			 strlen(reply_message),
			 reply_message);
    close(listen_socket);
    exit(-1);
  }
  
  /* Now myaddr_in contains the port and the internet address this is */
  /* returned to the sender also implicitly telling the sender that the */
  /* socket buffer sizing has been done. */

  final_settings->local_port_num = ntohs(myaddr_in.sin_port);
  strcpy(final_settings->test_dest,inet_ntoa(myaddr_in.sin_addr));

  sprintf(reply_message,
	  TCP_STREAM_RSP,
	  final_settings->local_socket_send,
	  final_settings->local_socket_recv,
	  final_settings->recv_size,
	  final_settings->local_port_num,
	  final_settings->local_nodelay,
	  final_settings->local_watermark,
	  settings->test_dest);

  send_control_message(test->control_sock,
		       TCP_STREAM,
		       strlen(reply_message),
		       reply_message);

  if ((data_socket = accept(listen_socket,
			    (struct sockaddr *)&peeraddr_in,
			    &addrlen)) == -1) {
    /* just punt - the remote should figure it out */
    close(listen_socket);
    exit(-1);
  }
  
  printf("Just accepted the data connection \n");

  /* where should this really go - just after the
     send_control_message, or just after the accept? */

  cpu_start(test);

  results->bytes_received = 0;
  results->num_receives = 0;

  while ((len = recv(data_socket,
		     recv_ring->buffer_ptr,
		     final_settings->recv_size,
		     0)) != 0) {
    if (len < 0) {
      /* the remote should get the clue when we toast the data
	 connection */
      exit(-1);
    }

    /* increament the bytes_received and number of recv_calls */
    results->bytes_received += len;
    results->num_receives++;

    /* move onto the next buffer in the ring */
    recv_ring = recv_ring->next;
  }
  
  /* at this point, we have gotten a recv of zero bytes, which means
     that the sender has closed his half of the connection. we now
     call shutdown on our half of the connection in response */

  if (shutdown(data_socket,1) == -1) {
    fprintf(where,
	    "Error in data socket shutdown - errno %d\n",errno);
    fflush(where);
    exit(-1);
  }

  cpu_stop(test);

  close(data_socket);
  close(listen_socket);
  if (debug) {
    dump_bsd_results(test);
  }
}

void
print_tcp_stream(test_t *test)
{

  double bytes_sent;
  double throughput;
  bsd_test_results *results;
  bsd_test_settings *final_settings;

  results = (bsd_test_results *)&(test->test_specific_results);
  final_settings = (bsd_test_settings *)&(results->final_settings);


  if (debug > 1) {
    dump_bsd_results(test);
  }

  bytes_sent = 
    (double)results->num_sends * (double)final_settings->send_size;

  if (debug) {
    fprintf(where,
	    "num_sends %d size %d \n",
	    results->num_sends,
	    final_settings->send_size);
    fflush(where);
  }

  throughput = calc_thruput(test,bytes_sent);

  printf("Thread %d elapsed time %f format %c tput %f\n",
	 test->thread_num,
	 test->elapsed_time,
	 test->format_units,
	 throughput);

}

void
send_tcp_rr(test_t *test)
{
}

void
recv_tcp_rr(test_t *test)
{
}

void
print_tcp_rr(test_t *test)
{
#ifdef DO_TCP_RR
  bsd_test_results *results;
  bsd_test_settings *final_settings;

  results = (bsd_test_results *)&(test->test_specific_results);
  final_settings = (bsd_test_settings *)&(results->final_settings);


  if (debug > 1) {
    dump_bsd_results(test);
  }

  if (debug) {
    fprintf(where,
	    "num_trans %d req size %d rsp size %d\n",
	    results->num_trans,
	    final_settings->req_size,
	    final_settings->rsp_size);
    fflush(where);
  }

  /* just to be sure that the format units are right */
  test->format_units = 't';

  throughput = calc_thruput(test,results->num_trans);

  printf("Thread %d elapsed time %f format %c tput %f\n",
	 test->thread_num,
	 test->elapsed_time,
	 test->format_units,
	 throughput);
#endif /* DO_TCP_RR */
}

void
nettest3_bsd_init()
{
  /* stick our function pointers into the apropriate places in the
     netperf_switch */  

  if (debug) {
    fprintf(where,"Initializing the BSD Tests with test %d\n",send_tcp_stream);
    fflush(where);
  }

  netperf_switch[TCP_STREAM][SCAN] = scan_bsd_args;
  netperf_switch[TCP_STREAM][SEND] = send_tcp_stream;
  netperf_switch[TCP_STREAM][RECV] = recv_tcp_stream;
  netperf_switch[TCP_STREAM][PRINT] = print_tcp_stream;

  netperf_switch[TCP_RR][SCAN] = scan_bsd_args;
  netperf_switch[TCP_RR][SEND] = send_tcp_rr;
  netperf_switch[TCP_RR][RECV] = recv_tcp_rr;
  netperf_switch[TCP_RR][PRINT] = print_tcp_rr;

}




