/*
 
              Copyright (C) 1999 Hewlett-Packard Company
                         ALL RIGHTS RESERVED.
 
  The enclosed software and documentation includes copyrighted works
  of Hewlett-Packard Co. For as long as you comply with the following
  limitations, you are hereby authorized to (i) use, reproduce, and
  modify the software and documentation, and to (ii) distribute the
  software and documentation, including modifications, for
  non-commercial purposes only.
      
  1.  The enclosed software and documentation is made available at no
      charge in order to advance the general development of
      high-performance networking products.
 
  2.  You may not delete any copyright notices contained in the
      software or documentation. All hard copies, and copies in
      source code or object code form, of the software or
      documentation (including modifications) must contain at least
      one of the copyright notices.
 
  3.  The enclosed software and documentation has not been subjected
      to testing and quality control and is not a Hewlett-Packard Co.
      product. At a future time, Hewlett-Packard Co. may or may not
      offer a version of the software and documentation as a product.
  
  4.  THE SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS".
      HEWLETT-PACKARD COMPANY DOES NOT WARRANT THAT THE USE,
      REPRODUCTION, MODIFICATION OR DISTRIBUTION OF THE SOFTWARE OR
      DOCUMENTATION WILL NOT INFRINGE A THIRD PARTY'S INTELLECTUAL
      PROPERTY RIGHTS. HP DOES NOT WARRANT THAT THE SOFTWARE OR
      DOCUMENTATION IS ERROR FREE. HP DISCLAIMS ALL WARRANTIES,
      EXPRESS AND IMPLIED, WITH REGARD TO THE SOFTWARE AND THE
      DOCUMENTATION. HP SPECIFICALLY DISCLAIMS ALL WARRANTIES OF
      MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
  
  5.  HEWLETT-PACKARD COMPANY WILL NOT IN ANY EVENT BE LIABLE FOR ANY
      DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
      (INCLUDING LOST PROFITS) RELATED TO ANY USE, REPRODUCTION,
      MODIFICATION, OR DISTRIBUTION OF THE SOFTWARE OR DOCUMENTATION.
 
*/

/* this utility is intended to parse a logfile in wu-ftpd command
   logging format into a format usable by the netperf FTP_DOWNLOAD
   test. it will separate out all the different sessions, and cull
   those which contain commands the benchmark does not support. 

   The output will be a series of discrete FTP sessions, ordered by
   their starting times. This will indeed be slightly different from
   the log itself, but is necessary because the netperf FTP_DOWNLOAD
   test does not know how to handle more than one session at a time. */

#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <strings.h>
#include "netlib3.h"
#include "nettest3_ftp.h"

char usage_string[] = "\
parse_logfile [-h] [-i input] [-w human_output] [-W machine_output]\n\
              [-v verbosity] [-a start_after] [-e end_before]\n";

typedef struct command {
  struct command *next;
  int    pad_bytes;      /* by how many bytes will the machine
			    readable version be padded */
  int    next_offset;
  int    command_code;
  int    retcode;
  char   *parm;
} command_t;

typedef struct ftp_session {
  struct ftp_session *next; /* next in the linked-list of sessions */
  struct ftp_session *hash; /* next in the hash chain of sessions */
  int    session_id;
  int    has_user;          /* true if we see a "USER" command */
  int    has_close;         /* true if we see some close indication */
  int    valid;             /* true unless we see something we do not
			       understand in this session */
  int    has_conn;          /* used to know if we have seen a PORT or
			       PASV prior to a RETR, LIST or
			       NLST. this is to capture bogosity with
			       syslog reordering logging messages */
  time_t start_time;        /* set to "curr_log_time" when we allocate
			       the session structure */
  time_t end_time;          /* updated to "curr_log_time" when we add
			       a parsed command*/
  time_t prev_log_time;     /* used to decide on think time */
  int    previous_command;  /* used to decide to think time */
  int    num_commands;      /* how many commands are attached to this
			       session? */
  char   *sut;              /* where is this session going? */
  command_t *head_command;          /* the head of the chain of
				       commands for this session */
  command_t *tail_command;          /* the tail of the chain of
				       commands for this session */ 
} ftp_session_t;

#define SESSION_HASH 1024
ftp_session_t session_hash[SESSION_HASH];

/* this is the global chain of sessions, which will be in the order of
   first appearance in the logfile */
ftp_session_t *head_session = NULL;
ftp_session_t *tail_session  = NULL;

int verbosity = 1;



/* allocate and initialize a new command structure, or die trying. */
command_t *
new_command()
{
  command_t *temp;

  temp = (command_t *) malloc(sizeof(command_t));
  if (temp) {
    temp->next = NULL;
    temp->next_offset = 0;
    temp->pad_bytes = 0;
    temp->command_code = OPEN;
    temp->retcode = 0;
    temp->parm = NULL;
  }
  else {
    fprintf(stderr,"unable to allocate new command structure\n");
    fflush(stderr);
    exit(-1);
  }
}

/* add a new command to an existing session */
void
add_command(ftp_session_t *session,
	    int command_code,
	    int retcode,
	    char *parm,
	    time_t command_time)
{
  command_t *temp;
  
  temp = new_command();

  session->num_commands++;
  session->prev_log_time = command_time;
  session->end_time = command_time;
  session->previous_command = temp->command_code = command_code;
  temp->retcode = retcode;
  temp->pad_bytes = 0;
  temp->next_offset = 0;

  if (parm) {
    temp->parm = strdup(parm);
    /* now, just to be sure */
    if (temp->parm == NULL) {
      fprintf(stderr,"could not strdup parm in add_command");
      fflush(stderr);
      exit(-1);
    }
    /* we have a parm, it will be a string, and we will want to add
       one for the null. at least I think we do :) raj 3/99 */
    temp->next_offset += strlen(parm) + 1;
  }

  /* a command in the workload file will have three ints, the command
     code, the retcode, and the next offset field itself. */
  temp->next_offset += (3 * sizeof(int));

  /* we want to make sure that we will start at the right boundary for
     ints and things in structs. */
  if ((temp->next_offset % sizeof(int))) {
    temp->pad_bytes = 4 - (temp->next_offset % sizeof(int));
  }
  temp->next_offset += temp->pad_bytes;

  /* just a really paranoid check, that should probably become an
     ASSERT or something */
  if (session->head_command == NULL) {
    fprintf(stderr,"sessions head command NULL in add_command\n");
    fflush(stderr);
    exit(-1);
  }
  session->tail_command->next = temp;
  session->tail_command = temp;
}



/* return a pointer to a new session structure, with its next command */
ftp_session_t *
new_session(int id, char *sut, int newok, time_t start_time)
{

  ftp_session_t *temp;
  int temp_offset;

  /* malloc a new session */
  temp = (ftp_session_t *)malloc(sizeof(ftp_session_t));

  if (temp) {
    temp->start_time = start_time;
    temp->has_user = newok;
    temp->has_close = 0;
    temp->has_conn = 0;
    temp->valid = 1;
    temp->next = temp->hash = NULL;
    temp->session_id = id;
    temp->prev_log_time = (time_t)0;
    temp->previous_command = OPEN; /* this will help avoid a WAIT on
				      the first command in the chain
				      */
    temp->sut = strdup(sut);
    temp->head_command = temp->tail_command = new_command();
    if 	((temp->head_command->parm = strdup(sut)) != NULL) {
      /* in the workload file, there will be three ints - next offset,
	 command code, and expected retcode, followed by an optional
	 parm. there is at least one NULL byte, and then sufficient
	 padding to take us to a an integer's alignment. */
      temp_offset = 3 * sizeof(int) + strlen(sut) + 1;
      if (temp_offset % sizeof(int)) {
	temp->head_command->pad_bytes = 4 - (temp_offset % sizeof(int));
      }
      temp_offset += temp->head_command->pad_bytes;
      temp->head_command->next_offset = temp_offset;
      temp->num_commands = 1;
#ifdef DEBUG
      printf("setting next_offset to %d\n",
	     temp->head_command->next_offset);
#endif /* DEBUG */

      return(temp);
    }
    else {
      fprintf(stderr,
	      "unable to allocate command struct or assign sut name\n");
      fflush(stderr);
      exit(-1);
    }
  }
  else {
    fprintf(stderr,"unable to allocate session struct.\n");
    fflush(stderr);
    exit(-1);
  }

}


/* this will return a pointer to an existing session, or will allocate
   a new session. if a new session is allocated, and new_ok is one,
   later that session will not be ignored. new_ok will only be set to
   one if we are calling get_session upon parsing a "USER" command
   from the ftpd logfile. any other command, and get_session will be
   called with new_ok equal to zero. raj 3/99 */
ftp_session_t *
get_session(int id, char *sut, int newok, time_t start_time)
{
  int hash;
  ftp_session_t *temp,*prev;

  hash = id % SESSION_HASH;
  temp = (ftp_session_t *)&(session_hash[hash]);

  /* walk the hash chain looking for our session id */
  while ((temp) && temp->session_id != id) {
    prev = temp;
    temp = temp->hash;
  }

  if (temp != NULL) {
    /* we much have made a match, or we would have walked until a NULL
     */
    return(temp);
  }
  else {
    /* we reached the end of the hash chain, allocate a new session */
    temp = new_session(id,sut,newok,start_time);
    if (temp) {
      /* join this new session to the hash chain */
      prev->hash = temp;

      /* join this new session to the list of sessions */
      if (head_session == NULL) {
	/* this is our first allocated after initializing */
	head_session = tail_session = temp;
      }
      else {
	tail_session->next = temp;
	tail_session = temp;
      }

      return temp;
    }
    else {
      fprintf(stderr,"could not allocate session. exiting\n");
      fflush(stderr);
      exit(-1);
    }
  }
}




char *
code_to_str(int command_code)
{
  switch (command_code) {
  case CLOS:
    return "CLOS";

  case CWD:
    return "CWD";
      
  case LIST:
    return "LIST";

  case MDTM:
    return "MDTM";

  case NLST:
    return "NLST";

  case NOOP:
    return "NOOP";

  case OPEN:
    return "OPEN";

  case PASS:
    return "PASS";

  case PASV:
    return "PASV";

  case PORT:
    return "PORT";

  case PWD:
    return "PWD";

  case QUIT:
    return "QUIT";

  case REST:
    return "REST";

  case RETR:
    return "RETR";

  case SIZE:
    return "SIZE";

  case STAT:
    return "STAT";

  case SYST:
    return "SYST";

  case TYPE:
    return "TYPE";

  case USER:
    return "USER";

  case WAIT:
    return "WAIT";

  default:
    return "UNKN";
  }
}




void
print_session(ftp_session_t *session,
	      FILE *human,
	      FILE *machine)
{
  command_t *temp;

  temp = session->head_command;

  while (temp) {
    if (human != NULL) {
      if (temp->parm) {
	fprintf(human,
		"%d %s %d %s\n",
		temp->next_offset,
		code_to_str(temp->command_code),
		temp->retcode,
		temp->parm);
      }
      else {
	fprintf(human,
		"%d %s %d\n",
		temp->next_offset,
		code_to_str(temp->command_code),
		temp->retcode);
      }
    }
    if (machine != NULL) {
      int nulls=0;
      fwrite(&(temp->next_offset), 
	     sizeof(temp->next_offset), 
	     1,
	     machine);
      fwrite(&(temp->command_code), 
	     sizeof(temp->command_code), 
	     1,
	     machine);
      fwrite(&(temp->retcode), 
	     sizeof(temp->retcode), 
	     1,
	     machine);
      /* was there a parm? */
      if (temp->parm) {
	fwrite(temp->parm,
	       sizeof(char),
	       strlen(temp->parm),
	       machine);
	/* dont forget to null-terminate the string */
	fwrite(&nulls,1,1,machine);
      }
      
      /* is there a needed padding? */
      if (temp->pad_bytes) {
	fwrite(&nulls,1,temp->pad_bytes,machine);
      }
    }
    temp = temp->next;
  }
}

void
skip_command(char *command, ftp_session_t *session, int line, char *reason)
{
  if (verbosity > 1) {
    fprintf(stderr,
	    "Session %d invalidated at line %d for command %s",
	    session->session_id,
	    line,
	    command);
    if (reason) {
      fprintf(stderr,
	      " : %s\n",reason);
    }
    else {
      fprintf(stderr,"\n");
    }
    fflush(stderr);
  }
  session->valid = 0;
   
}



int
main(int argc, char *argv[])

{
  FILE *input = stdin;
  FILE *human = stdout;
  FILE *machine = NULL;

  int valid_sessions,invalid_sessions;

  char line[BUFSIZ];

  struct tm curr_log_date;
  struct tm temp_date;

  time_t    curr_log_time,prev_log_time;
  char datestr[BUFSIZ];

  time_t start_after = (time_t)0;
  time_t end_before = (time_t)0;

  ftp_session_t *curr_session;

  int error;
  int i;
  int time_delta;
  int current_line;

  int32_t last_command = WAIT;

  char c;

  for(i = 0; i < SESSION_HASH; i++) {
    session_hash[i].has_user = 0;
    session_hash[i].next = NULL;
    session_hash[i].hash = NULL;
    session_hash[i].session_id = -1 * i;
    session_hash[i].prev_log_time = (time_t)0;
    session_hash[i].previous_command = OPEN; /* this will help avoid a WAIT on
				      the first command in the chain
				      */
    session_hash[i].sut = strdup("1_2_3_4_5");
    session_hash[i].head_command = NULL;
    session_hash[i].tail_command = NULL;
  }

  while ((c = getopt(argc,argv,"a:e:hi:v:w:W:")) != -1) {
  switch (c) {

  case 'a':
    if (getdate_r(optarg,&temp_date,&error) == -1) {
      fprintf(stderr,
	      "getdate_r returned error %d for start_after\n",
	      error);
      fflush(stderr);
      exit(-1);
    }
    start_after = mktime(&temp_date);
    break;

  case 'e':
    if (getdate_r(optarg,&temp_date,&error) == -1) {
      fprintf(stderr,
	      "getdate_r returned error %d for end_before\n",
	      error);
      fflush(stderr);
      exit(-1);
    }
    end_before = mktime(&temp_date);
    break;

  case 'i':
    input = fopen(optarg,"r");
    if (input == NULL) {
      perror("could not open input file for reading");
      exit(-1);
    }
    break;

  case 'v':
    verbosity = atoi(optarg);
    break;

  case 'w':
    human = fopen(optarg,"w");
    if (human == NULL) {
      perror("could not open human-readable output file for writing");
      exit(-2);
    }
    break;

  case 'W':
    machine = fopen(optarg,"w");
    if (machine == NULL) {
      perror("could not open human-readable output file for writing");
      exit(-2);
    }
    break;

  case 'h':
  case '?':
  case ':':
    fprintf(stderr,usage_string);
    exit(-1);
  }
  }

  prev_log_time = (time_t)0;

  current_line = 0;
  while (fgets(line, BUFSIZ, input) != NULL) {
    char *month,*day,*daytime;
    char *sut,*daemon,*id;
    char *command,*parm;

    current_line++;

    /* if the logfile does not have the format we assume, this will
       probably get hopelessly confused. raj 3/99 */
    month     = strtok(line," ");
    day       = strtok(NULL," ");
    daytime   = strtok(NULL," ");
    sut       = strtok(NULL," ");
    daemon    = strtok(NULL," []:");
    id        = strtok(NULL,"[]:");
    command   = strtok(NULL,"]: \n");
    parm      = strtok(NULL," \n");

    if ((month) &&
	(day) &&
	(daytime)) {
      sprintf(datestr,"%s %s %s",month,day,daytime);
    }
    else {
      fprintf(stderr,
	      "No date information in logfile! Aborting at line %d.\n",
	      current_line);
      fflush(stderr);
      exit(-1);
    }

    if (getdate_r(datestr,&curr_log_date,&error) == -1) {
      fprintf(stderr,"getdate_r returned error %d\n",error);
      fflush(stderr);
      exit(-1);
    }

    curr_log_time = mktime(&curr_log_date);

    /* I'm not sure if we need to check for a change in the daemon
       name, or the sut name and punt if that occurs, but for some
       sort of multi-homed server that could be an issue that we will
       need to deal with. raj 3/99 */

    if ((sut) && 
	(id) &&
	(daemon) &&
	(command)) {
      /* the parsing seems to be OK so far, so grab a session. only
	 mark it as has_user if the current command is "USER" */
      if (strcmp("USER",command) == 0) {
	curr_session = get_session(atoi(id),
				   sut,
				   1,
				   curr_log_time);
      }
      else {
	curr_session = get_session(atoi(id),
				   sut,
				   0,
				   curr_log_time);
      }
    }
    else {
      fprintf(stderr,
	      "could not parse sut, id, daemon, or command. exiting\n");
      fflush(stderr);
      exit(-1);
    }

    /* now we try to guesstimate the user's think time. we do this by
       comparing the timestamp of the previous command fr this session
       with the timestamp for the current command. if the difference
       is greater than or equal to two seconds, we use the difference
       less one second as a think time. we do not use a difference of
       one second or greater because we have no idea if that was
       simply a roll-over on the timestamp - which only has accuracy
       to a second.

       also, we do not add a think time command if the prior command
       for this session was either a WAIT command, or a command that
       involves some non-trivial quantity of data transfer -
       basically, anything that involves opening a data connection. we
       ignore these time deltas because we have no idea what the size
       of the transfer was, nor the transfer rate happened to be to
       guess when the transfer actually would have finished prior to
       the next command.

       the one current exception to this is dealt with later when we
       encounter a "User" "timed out after" command - when we parse
       that one, we introduce a WAIT command regardless of the prior
       command - the "User timed out after" line tells us explicitly
       how long the session sat idle so we can go ahead and add it.

       there is a slight problem with the heruistic of looking at the
       time deltas when the session is from a system that is far away
       - sometimes, it could simply be greater than one or two seconds
       just to have a command complete - i have seen some instances of
       sessions from across the globe having delays within a series of
       STAT commands, that were almost certainly not
       human-induced. one of these days, I suppose it would be
       necessary to be a triffle more intelligent about deciding on
       inserting WAIT commands. for now, we will simply call it a
       crude simulation of WAN and packet loss delays :)

       raj 3/99 */

    time_delta = curr_log_time - (curr_session->prev_log_time);
    if ((time_delta >= 2) &&
	(curr_session->previous_command != WAIT) &&
	(curr_session->previous_command != RETR) &&
	(curr_session->previous_command != LIST) &&
	(curr_session->previous_command != OPEN) &&
	(curr_session->previous_command != NLST)) {
      char temp_wait_time[64];
      sprintf(temp_wait_time,
	      "%d",
	      (time_delta - 1));
      add_command(curr_session, 
		  WAIT, 
		  0,
		  temp_wait_time,
		  curr_log_time);
    }
    
    /* we would like these to be more or less in frequency order */
    if (strcmp(command,"CWD") == 0) {
      if (parm) {
	add_command(curr_session,CWD,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"TYPE") == 0) {
      if (parm) {
	add_command(curr_session,TYPE,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"PASV") == 0) {
      if (curr_session->has_conn == 0) {
	add_command(curr_session,PASV,0,NULL,curr_log_time);
	curr_session->has_conn = 1;
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "duplicate PASV/PORT");
      }
    }
    else if (strcmp(command,"USER") == 0) {
      if (parm) {
	add_command(curr_session,USER,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"PASS") == 0) {
      if (parm) {
	add_command(curr_session,PASS,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"LIST") == 0) {
      if (curr_session->has_conn == 1) {
	if (parm) {
	  add_command(curr_session,LIST,0,parm,curr_log_time);
	}
	else {
	  add_command(curr_session,LIST,0,NULL,curr_log_time);
	}
	curr_session->has_conn=0;
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing PORT/PASV");
      }
    }
    else if (strcmp(command,"RETR") == 0) {
      if (curr_session->has_conn == 1) {
	if (parm) {
	  add_command(curr_session,RETR,0,parm,curr_log_time);
	  curr_session->has_conn=0;
	}
	else {
	  skip_command(command,
		       curr_session,
		       current_line,
		       "missing parameter");
	}
      }
      else {
	  skip_command(command,
		       curr_session,
		       current_line,
		       "missing PASV/PORT");
      }
    }
    else if (strcmp(command,"STAT") == 0) {
      if (parm) {
	add_command(curr_session,STAT,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"SIZE") == 0) {
      if (parm) {
	add_command(curr_session,SIZE,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"PORT") == 0) {
      if (curr_session->has_conn == 0) {
	add_command(curr_session,PORT,0,NULL,curr_log_time);
	curr_session->has_conn = 1;
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "duplicate PASV/PORT");
      }
    }
    else if (strcmp(command,"MDTM") == 0) {
      if (parm) {
	add_command(curr_session,MDTM,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"SYST") == 0) {
      add_command(curr_session,SYST,0,NULL,curr_log_time);
    }
    else if (strcmp(command,"QUIT") == 0) {
      curr_session->has_close = 1;
      add_command(curr_session,QUIT,0,NULL,curr_log_time);
    }
    else if (strcmp(command,"REST") == 0) {
      if (parm) {
	add_command(curr_session,REST,0,parm,curr_log_time);
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing parameter");
      }
    }
    else if (strcmp(command,"PWD") == 0) {
      add_command(curr_session,PWD,0,NULL,curr_log_time);
    }
    else if (strcmp(command,"NLST") == 0) {
      if (curr_session->has_conn == 1) {
	curr_session->has_conn = 0;
	if (parm) {
	  add_command(curr_session,NLST,0,parm,curr_log_time);
	}
	else {
	  add_command(curr_session,NLST,0,NULL,curr_log_time);
	}
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "missing PORT/PASV");
      }
    }
    else if (strcmp(command,"NOOP") == 0) {
      add_command(curr_session,NOOP,0,NULL,curr_log_time);
    }
    
    else if (strcmp(command,"FTP") == 0) {
      char *closed;
      closed = strtok(NULL," \n");
      if ((parm != NULL) && 
	  (strcmp(parm,"session") == 0) &&
	  (closed != NULL) && 
	  (strcmp(closed,"closed") == 0)) {
	/* the session has ended, so add a CLOS command */
	add_command(curr_session,CLOS,0,NULL,curr_log_time);
	curr_session->has_close = 1;
      }
      else {
	/* not quite what this is, so lets ignore the session */
	skip_command(command,
		     curr_session,
		     current_line,
		     "unfamiliar parameter");
      }
    }
    else if (strcmp(command,"ANONYMOUS") == 0) {
      char *foo;
      foo = strtok(NULL," ");
      if ((parm) && 
	  (strcmp(parm,"FTP") == 0) &&
	  (foo) && 
	  (strcmp(foo,"LOGIN") == 0)) {
      }
      else {
	/* not quite what this is, so lets ignore the session */
	skip_command(command,
		     curr_session,
		     current_line,
		     "unfamiliar parameter");
      }
    }
    else if (strcmp(command,"User") == 0) {
      char *timed;
      char *out;
      char *after;
      char *timeout;
      timed   = strtok(NULL," ");
      out     = strtok(NULL," ");
      after   = strtok(NULL," ");
      timeout = strtok(NULL," ");

      /* this is going to be ugly, there is undoubtedly a better way
	 to do this... */
      if ((parm) &&
	  (strcmp(parm,"ftp") == 0) &&
	  (timed) &&
	  (strcmp(timed,"timed") == 0) &&
	  (out) &&
	  (strcmp(out,"out") == 0) &&
	  (after) &&
	  (strcmp(after,"after") == 0) &&
	  (timeout)) {
	if (curr_session->previous_command != WAIT) {
	/* the code above us should have resulted in the addition of a
	   WAIT command, so there really is nothing else to do with
	   the "User ftp timed out..." message. however, sometimes,
	   the command above might have been one that involved data
	   transfer, so we may need to add a WAIT command here
	   directly since this line being parse is a direct statement
	   of idleness on the connection regardless of prior
	   command. raj 3/99 */ 
	add_command(curr_session,WAIT,0,timeout,curr_log_time);
	}
	else {
	  /* silently ignore */
	}
      }
      else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "unfamiliar parameter");
      }
    }
    else {
	skip_command(command,
		     curr_session,
		     current_line,
		     "unsupported command");
    }

  }

  /* we have finished going through the entire logfile, how many
     sessions do we have? */

  curr_session = head_session;
  
  valid_sessions = invalid_sessions = 0;
  while (curr_session) {
    if (verbosity > 2) {
      printf("session %d commands %d has_user %d has_close %d valid %d\n",
	     curr_session->session_id,
	     curr_session->num_commands,
	     curr_session->has_user,
	     curr_session->has_close,
	     curr_session->valid);
    }
    if ((curr_session->has_user) &&
	(curr_session->has_close) &&
	(curr_session->valid) &&
	((start_after == 0) || (start_after <=
				curr_session->start_time)) &&
	((end_before == 0)  || (end_before  >= 
				curr_session->end_time))) {
      valid_sessions++;
      if (verbosity >= 1) {
	print_session(curr_session,human,machine);
      }
    }
    else {
      invalid_sessions++;
    }
    curr_session = curr_session->next;
  }
    
  fprintf(stderr,
	  "Selected %d sessions out of %d (%d skipped)\n",
	  valid_sessions,
	  valid_sessions + invalid_sessions,
	  invalid_sessions);
  fflush(stderr);

  return 0;
}
