/*
 *	SILCConnectionHandler.java		2002/11/16
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.util.Arrays;
import com.alistairphipps.util.Hex;
import java.lang.String;
import com.alistairphipps.jsilc.core.*;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;

/** Handles translation between SILC packets and events and vice versa
 * 
 * @author Alistair K Phipps
 * @version 20021117
 */
public class SILCConnectionHandler implements ConnectionHandler
{
	private SILCConnection _conn;
	
	public SILCConnectionHandler( String strUsername, String strHost, int iPort ) throws java.net.UnknownHostException, java.io.IOException
	{
		_conn = new SILCConnection( strUsername, strHost, iPort );
	}
	
	public synchronized void putEvent( OutEvent e )
	{
		try
		{
			if( e instanceof MessageOutEvent )
			{
				SILCPayload pl = new SILCGenericMessagePayload( false, true, ((MessageOutEvent)e).getMessage() );	// false as not an action, true as UTF8 encoded
				SILCPacket p = new SILCPacket( SILCPacketType.PRIVATE_MESSAGE, _conn.getLocalIdType(), _conn.getLocalId(), ( (SILCEntity)e.getDest() ).getIdType(), ( (SILCEntity)e.getDest() ).getId(), pl );
				_conn.putPacket( p );
			} // TODO: handle other types of events
		}
		catch( Exception x )
		{
			x.printStackTrace();
		}
	}

	public InEvent getEvent()
	{
		try
		{
			SILCPacket p = _conn.getPacket();
			MessageInEvent e;
			switch( p.getType() )
			{
			case SILCPacketType.DISCONNECT:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Disconnect: \n" + p.toString() );			
				break;
			case SILCPacketType.SUCCESS:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Success: \n" + p.toString() );			
				break;
			case SILCPacketType.FAILURE:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Failure: \n" + p.toString() );			
				break;
			case SILCPacketType.REJECT:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Reject: \n" + p.toString() );			
				break;
			case SILCPacketType.NOTIFY:
				// e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Notify: \n" + ((SILCNotifyPayload)p.getPayload() ).toString() );			
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), ( (SILCNotifyPayload)p.getPayload() ).getArgumentData() );
				break;
			case SILCPacketType.ERROR:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Error: \n" + p.toString() );			
				break;
			case SILCPacketType.CHANNEL_MESSAGE:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Channel Message: \n" + p.toString() );			
				break;
			case SILCPacketType.CHANNEL_KEY:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Channel Key: \n" + p.toString() );			
				break;
			case SILCPacketType.PRIVATE_MESSAGE:
				System.out.println(p.toString());	
				e = new MessageInEvent( new SILCEntity( p.getSourceIdType(), p.getSourceId(), getFriendlyName( p.getSourceIdType(), p.getSourceId() ) ), ( (SILCGenericMessagePayload)p.getPayload() ).getMessage() );
				break;
			case SILCPacketType.PRIVATE_MESSAGE_KEY:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Private Message Key: \n" + p.toString() );			
				break;
			case SILCPacketType.COMMAND:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Command: \n" + p.toString() );			
				break;
			case SILCPacketType.COMMAND_REPLY:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Command Reply: \n" + p.toString() );			
				break;
			case SILCPacketType.KEY_EXCHANGE:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Key Exchange: \n" + p.toString() );			
				break;
			case SILCPacketType.KEY_EXCHANGE_1:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Key Exchange 1: \n" + p.toString() );			
				break;
			case SILCPacketType.KEY_EXCHANGE_2:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Key Exchange 2: \n" + p.toString() );			
				break;
			case SILCPacketType.CONNECTION_AUTH_REQUEST:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Connection Auth Request: \n" + p.toString() );			
				break;
			case SILCPacketType.CONNECTION_AUTH:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Connection Auth: \n" + p.toString() );			
				break;
			case SILCPacketType.NEW_ID:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received New Id: \n" + p.toString() );			
				break;
			case SILCPacketType.HEARTBEAT:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Received Heartbeat" );
				break;
			
			default:
				e = new MessageInEvent( new SILCEntity( _conn.getRemoteIdType(), _conn.getRemoteId(), getFriendlyName( _conn.getRemoteIdType(), _conn.getRemoteId() ) ), "Unknown SILC packet received: \n" + p.toString() );
			}
			return e;
		}
		catch( Exception x )
		{
			x.printStackTrace();
			return null; // XXX maybe return an informative message instead
		}
	}

	private String getFriendlyName( byte yIdType, byte[] ylId )
	{
		String strRet = SILCIdType.toString( yIdType ) + " - ";
		if( ( yIdType == _conn.getRemoteIdType() ) && ( Arrays.equals( ylId, _conn.getRemoteId() ) ) )
		{
			return strRet + _conn.getServerName();
		}
		if( ( yIdType == SILCIdType.CLIENT ) )
		{
			return strRet + getClientName( yIdType, ylId );
		}
		return strRet + "Unknown";
	}

	private String getClientName( byte yIdType, byte[] ylId )
	{	// 
		String strRet = Hex.toString( ylId );	// fallback
		try	// FIXME: change packet constructing and easier Argument Payload checking
		{
			// send a identify
			SILCArgumentPayload[] argl = new SILCArgumentPayload[ 1 ];
			
			ByteBuffer yb = ByteBuffer.allocate( 4 + ylId.length );
			yb.putShort( (short)( yIdType & 0xFF ) );
			yb.putShort( (short)ylId.length );
			yb.put( ylId );

			argl[0] = new SILCArgumentPayload( (byte)5, yb.array());
			
			SILCPayload pl = new SILCCommandPayload( SILCCommandType.IDENTIFY, argl );
			SILCPacket p = new SILCPacket( SILCPacketType.COMMAND, _conn.getLocalIdType(), _conn.getLocalId(), _conn.getRemoteIdType(), _conn.getRemoteId(), pl );
			_conn.putPacket( p );

			// wait for whois response
			SILCPacket pin = _conn.getPacket();
			//assert( pin.getType() == SILCPacketType.COMMAND_REPLY ); // FIXME: shouldn't assert - we could get a failure or whatever
			if (pin.getType() == SILCPacketType.COMMAND_REPLY)
			{
				SILCCommandPayload plin = (SILCCommandPayload)pin.getPayload();
				SILCArgumentPayload[] arglIn = plin.getArgList();
				Charset csUTF8 = Charset.forName( "UTF-8" );

				for( int i = 0; i < arglIn.length; i++ )
				{
					if( arglIn[i].getType() == 1 )
					{
						ByteBuffer temp = ByteBuffer.wrap( arglIn[i].getData());
						byte bStatus = temp.get();
						byte bError = temp.get();
						
						// System.out.println("Status: "+SILCStatusType.toString(bStatus)+" Error:"+SILCStatusType.toString(bError));
					}
				}
	
				for( int i = 0; i < arglIn.length; i++ )
				{
					if( arglIn[i].getType() == 3 )
					{
						strRet = csUTF8.decode( ByteBuffer.wrap( arglIn[i].getData() ) ).toString();
					}
				}
			}
		}
		catch( Exception e )
		{
			e.printStackTrace();
		}
		finally
		{
			return strRet;
		}
	}

	private byte[] getClientId( String strName )
	{
		
		String strRet = "";
		byte[] byteRet = strRet.getBytes();

		try	// FIXME: change packet constructing and easier Argument Payload checking
		{			
			// send a identify
			SILCArgumentPayload[] argl = new SILCArgumentPayload[ 1 ];
			
			Charset s_csUTF8 = Charset.forName( "UTF-8" );
			ByteBuffer yb = s_csUTF8.encode( strName );

			argl[0] = new SILCArgumentPayload( (byte)1, yb.array());
			
			SILCPayload pl = new SILCCommandPayload( SILCCommandType.IDENTIFY, argl );
			SILCPacket p = new SILCPacket( SILCPacketType.COMMAND, _conn.getLocalIdType(), _conn.getLocalId(), _conn.getRemoteIdType(), _conn.getRemoteId(), pl );
			_conn.putPacket( p );

			// wait for whois response
			SILCPacket pin = _conn.getPacket();
			//assert( pin.getType() == SILCPacketType.COMMAND_REPLY ); // FIXME: shouldn't assert - we could get a failure or whatever
			if (pin.getType() == SILCPacketType.COMMAND_REPLY)
			{
				SILCCommandPayload plin = (SILCCommandPayload)pin.getPayload();
				SILCArgumentPayload[] arglIn = plin.getArgList();

				for( int i = 0; i < arglIn.length; i++ )
				{
					if( arglIn[i].getType() == 1 )
					{
						ByteBuffer temp = ByteBuffer.wrap( arglIn[i].getData());
						byte bStatus = temp.get();
						byte bError = temp.get();
						
						System.out.println("Status: "+SILCStatusType.toString(bStatus)+" Error:"+SILCStatusType.toString(bError));
					}
				}
	
				for( int i = 0; i < arglIn.length; i++ )
				{
					if( arglIn[i].getType() == 2 )
					{
						strRet = s_csUTF8.decode( ByteBuffer.wrap( arglIn[i].getData() ) ).toString();
						byteRet = strRet.getBytes();
					}
				}
			}
		}
		catch( Exception e )
		{
			e.printStackTrace();
		}
		finally
		{
			return byteRet;
		}

	}
}

