
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific functions for manipulating thread state.
                                                    vg_ppc_state.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "vg_include.h"
#include <string.h>
#include <errno.h>
#include <sys/ptrace.h>
#include <asm/ptrace.h>

/*====================================================================*/
/*=== baseBlock: definition + setup                                ===*/
/*====================================================================*/

/* The variables storing offsets. */

Int VGOFF_(m_gpr) = INVALID_OFFSET;
Int VGOFF_(m_esp) = INVALID_OFFSET;
Int VGOFF_(m_eip) = INVALID_OFFSET;
Int VGOFF_(m_cr) = INVALID_OFFSET;
Int VGOFF_(m_lr) = INVALID_OFFSET;
Int VGOFF_(m_ctr) = INVALID_OFFSET;
Int VGOFF_(m_xer) = INVALID_OFFSET;
Int VGOFF_(spillslots) = INVALID_OFFSET;
Int VGOFF_(sh_gpr) = INVALID_OFFSET;
Int VGOFF_(sh_esp) = INVALID_OFFSET;
Int VGOFF_(sh_cr) = INVALID_OFFSET;
Int VGOFF_(sh_lr) = INVALID_OFFSET;
Int VGOFF_(sh_ctr) = INVALID_OFFSET;
Int VGOFF_(sh_xer) = INVALID_OFFSET;

Int VGOFF_(fpu_state_ptr) = INVALID_OFFSET;
Int VGOFF_(vec_state_ptr) = INVALID_OFFSET;
Int VGOFF_(helper_loadfpu) = INVALID_OFFSET;
Int VGOFF_(helper_storefpu) = INVALID_OFFSET;
Int VGOFF_(helper_loadvec) = INVALID_OFFSET;
Int VGOFF_(helper_cache_inval) = INVALID_OFFSET;
Int VGOFF_(total_synth_instrs) = INVALID_OFFSET;
Int VGOFF_(total_real_instrs) = INVALID_OFFSET;

ULong VG_(total_synth_instrs);
ULong VG_(total_real_instrs);
Addr VG_(dispatch_sp);

void arch_alloc_low_baseBlock ( Addr client_eip, Addr esp_at_startup )
{
   /* 0  */	VGOFF_(m_gpr)	= VG_(alloc_BaB)(32);
		memset(&VG_(baseBlock)[VGOFF_(m_gpr)], 0, 32 * sizeof(UInt));
   /* 1  */	VGOFF_(m_esp)	= VGOFF_(m_gpr) + 1;
		VG_(baseBlock)[VGOFF_(m_esp)] = esp_at_startup;
   /* 32 */	VGOFF_(m_eip)	= VG_(alloc_BaB_1_set)(client_eip);
   /* 33 */	VGOFF_(m_cr)	= VG_(alloc_BaB_1_set)(0);
   /* 34 */	VGOFF_(m_lr)	= VG_(alloc_BaB_1_set)(0);
   /* 35 */	VGOFF_(m_ctr)	= VG_(alloc_BaB_1_set)(0);
   /* 36 */	VGOFF_(m_xer)	= VG_(alloc_BaB_1_set)(0);
   if (VG_(needs).shadow_regs) {
      /* 36 */	VGOFF_(sh_gpr)	= VG_(alloc_BaB)(32);
		memset(&VG_(baseBlock)[VGOFF_(sh_gpr)], 0, 32 * sizeof(UInt));
      /* 37 */	VGOFF_(sh_esp)	= VGOFF_(sh_gpr) + 1;
      /* 68 */	VGOFF_(sh_cr)	= VG_(alloc_BaB_1_set)(0);
      /* 69 */	VGOFF_(sh_lr)	= VG_(alloc_BaB_1_set)(0);
      /* 70 */	VGOFF_(sh_ctr)	= VG_(alloc_BaB_1_set)(0);
      /* 71 */	VGOFF_(sh_xer)	= VG_(alloc_BaB_1_set)(0);
      VG_TRACK( post_regs_write_init );
   }
}

void arch_alloc_baseBlock( Addr client_eip, Addr esp_at_startup )
{
   VGOFF_(spillslots) = VG_(alloc_BaB)(VG_MAX_SPILLSLOTS);

#  define HELPER(name) \
   VGOFF_(helper_##name) = VG_(alloc_BaB_1_set)( (Addr) & VG_(helper_##name))

   /* Helper functions. */
   HELPER(loadfpu);
   HELPER(storefpu);
   HELPER(loadvec);
   HELPER(cache_inval);

   VGOFF_(fpu_state_ptr) = VG_(alloc_BaB)(1);
   VGOFF_(vec_state_ptr) = VG_(alloc_BaB)(1);
   VGOFF_(total_synth_instrs) = VG_(alloc_BaB)(1);
   VGOFF_(total_real_instrs) = VG_(alloc_BaB)(1);

#  undef HELPER

}

/* Junk to fill up a thread's shadow regs with when shadow regs aren't
   being used. */
#define VG_UNUSED_SHADOW_REG_VALUE  0x27182818

void VG_(arch_load_state)( arch_thread_t *regs, ThreadId tid )
{
   Int i;

   for (i = 0; i < 32; ++i)
      VG_(baseBlock)[VGOFF_(m_gpr) + i] = regs->m_gpr[i];
   VG_(baseBlock)[VGOFF_(m_eip)] = regs->m_eip;
   VG_(baseBlock)[VGOFF_(m_ctr)] = regs->m_ctr;
   VG_(baseBlock)[VGOFF_(m_lr)]  = regs->m_lr;
   VG_(baseBlock)[VGOFF_(m_xer)] = regs->m_xer;
   VG_(baseBlock)[VGOFF_(m_cr)]  = regs->m_cr;
   if (VG_(needs).shadow_regs) {
      for (i = 0; i < 32; ++i)
	 VG_(baseBlock)[VGOFF_(sh_gpr) + i] = regs->sh_gpr[i];
      VG_(baseBlock)[VGOFF_(sh_ctr)] = regs->sh_ctr;
      VG_(baseBlock)[VGOFF_(sh_lr)]  = regs->sh_lr;
      VG_(baseBlock)[VGOFF_(sh_xer)] = regs->sh_xer;
      VG_(baseBlock)[VGOFF_(sh_cr)]  = regs->sh_cr;
   } else {
      /* Fields shouldn't be used -- check their values haven't changed. */
      for (i = 0; i < 32; ++i)
	 vg_assert(VG_UNUSED_SHADOW_REG_VALUE == regs->sh_gpr[i]);
      vg_assert(
         VG_UNUSED_SHADOW_REG_VALUE == regs->sh_ctr &&
         VG_UNUSED_SHADOW_REG_VALUE == regs->sh_lr  &&
         VG_UNUSED_SHADOW_REG_VALUE == regs->sh_xer &&
         VG_UNUSED_SHADOW_REG_VALUE == regs->sh_cr);
   }

   /* FPU and VR states are in the ThreadState at present */
   VG_(baseBlock)[VGOFF_(fpu_state_ptr)] = (Addr) &regs->m_fpr[0];
   if (VG_(hardware_capabilities) & VKI_HWCAP_ALTIVEC)
      VG_(baseBlock)[VGOFF_(vec_state_ptr)] = (Addr) &regs->m_vr[0];

   VG_(baseBlock)[VGOFF_(total_synth_instrs)] = 0;
   VG_(baseBlock)[VGOFF_(total_real_instrs)] = 0;
}

void VG_(arch_save_state)( arch_thread_t *regs, ThreadId tid )
{
   Int i;
   const UInt junk = 0xDEADBEEF;

   VG_(total_synth_instrs) += VG_(baseBlock)[VGOFF_(total_synth_instrs)];
   VG_(total_real_instrs) += VG_(baseBlock)[VGOFF_(total_real_instrs)];
   VG_(baseBlock)[VGOFF_(total_synth_instrs)] = 0;
   VG_(baseBlock)[VGOFF_(total_real_instrs)] = 0;

   for (i = 0; i < 32; ++i)
      regs->m_gpr[i] = VG_(baseBlock)[VGOFF_(m_gpr) + i];
   regs->m_ctr = VG_(baseBlock)[VGOFF_(m_ctr)];
   regs->m_lr  = VG_(baseBlock)[VGOFF_(m_lr)];
   regs->m_xer = VG_(baseBlock)[VGOFF_(m_xer)];
   regs->m_cr  = VG_(baseBlock)[VGOFF_(m_cr)];
   regs->m_eip = VG_(baseBlock)[VGOFF_(m_eip)];
   if (VG_(needs).shadow_regs) {
      for (i = 0; i < 32; ++i)
	 regs->sh_gpr[i] = VG_(baseBlock)[VGOFF_(sh_gpr) + i];
      regs->sh_ctr = VG_(baseBlock)[VGOFF_(sh_ctr)];
      regs->sh_lr  = VG_(baseBlock)[VGOFF_(sh_lr)];
      regs->sh_xer = VG_(baseBlock)[VGOFF_(sh_xer)];
      regs->sh_cr  = VG_(baseBlock)[VGOFF_(sh_cr)];
   } else {
      for (i = 0; i < 32; ++i)
	 regs->sh_gpr[i] = VG_UNUSED_SHADOW_REG_VALUE;
      regs->sh_ctr = VG_UNUSED_SHADOW_REG_VALUE;
      regs->sh_lr  = VG_UNUSED_SHADOW_REG_VALUE;
      regs->sh_xer = VG_UNUSED_SHADOW_REG_VALUE;
      regs->sh_cr  = VG_UNUSED_SHADOW_REG_VALUE;
   }

   for (i = 0; i < 32; ++i)
      VG_(baseBlock)[VGOFF_(m_gpr) + i] = junk;
   VG_(baseBlock)[VGOFF_(m_eip)] = junk;
   VG_(baseBlock)[VGOFF_(m_ctr)] = junk;
   VG_(baseBlock)[VGOFF_(m_lr)]  = junk;
   VG_(baseBlock)[VGOFF_(m_xer)] = junk;
   VG_(baseBlock)[VGOFF_(m_cr)]  = junk;

#if 0
   if (VG_(baseBlock)[VGOFF_(fpu_state_ptr)] == 0) {
      /* the thread loaded up the FPU; save it back to the thread state */
      VG_(save_fpu_state)(&regs->m_fpr[0]);
   }
#endif
   /* set this to dangerous nonsense in case anyone tries to use it */
   VG_(baseBlock)[VGOFF_(fpu_state_ptr)] = junk;

   if ((VG_(hardware_capabilities) & VKI_HWCAP_ALTIVEC)
       && VG_(baseBlock)[VGOFF_(vec_state_ptr)] == 0) {
      /* the thread loaded up the vector registers;
	 save it back to the thread state */
      VG_(save_vec_state)(&regs->m_vr[0]);
   }
   /* set this to dangerous nonsense in case anyone tries to use it */
   VG_(baseBlock)[VGOFF_(vec_state_ptr)] = junk;
}


void VG_(dump_BB_regs)(void)
{
   int i;

   if (VG_(needs).shadow_regs) {
      for (i = 0; i < 32; ++i) {
	 if ((i & 3) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x:%08x", VG_(baseBlock)[i],
		     VG_(baseBlock)[VGOFF_(sh_gpr) + i]);
	 if ((i & 3) == 3)
	    VG_(printf)("\n");
      }
      VG_(printf)("  lr: %08x:%08x  ctr: %08x:%08x\n",
		VG_(baseBlock)[VGOFF_(m_lr)], VG_(baseBlock)[VGOFF_(sh_lr)],
		VG_(baseBlock)[VGOFF_(m_ctr)], VG_(baseBlock)[VGOFF_(sh_ctr)]);
      VG_(printf)("  cr: %08x:%08x  xer: %08x:%08x\n",
		VG_(baseBlock)[VGOFF_(m_cr)], VG_(baseBlock)[VGOFF_(sh_cr)],
		VG_(baseBlock)[VGOFF_(m_xer)], VG_(baseBlock)[VGOFF_(sh_xer)]);
   } else {
      for (i = 0; i < 32; ++i) {
	 if ((i & 7) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x", VG_(baseBlock)[i]);
	 if ((i & 7) == 7)
	    VG_(printf)("\n");
      }
      VG_(printf)(" lr: %08x  ctr: %08x  cr: %08x  xer: %08x\n",
		  VG_(baseBlock)[VGOFF_(m_lr)], VG_(baseBlock)[VGOFF_(m_ctr)], 
		  VG_(baseBlock)[VGOFF_(m_cr)], VG_(baseBlock)[VGOFF_(m_xer)]);
   }
   VG_(printf)("NIP: %08x\n", VG_(baseBlock)[VGOFF_(m_eip)]);
}

void VG_(dump_thread_regs)(ThreadState *tst)
{
   int i;

   if (VG_(needs).shadow_regs) {
      for (i = 0; i < 32; ++i) {
	 if ((i & 3) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x:%08x", tst->arch.m_gpr[i], tst->arch.sh_gpr[i]);
	 if ((i & 3) == 3)
	    VG_(printf)("\n");
      }
      VG_(printf)(" lr: %08x:%08x  ctr: %08x:%08x\n",
		  tst->arch.m_lr, tst->arch.sh_lr,
		  tst->arch.m_ctr, tst->arch.sh_ctr);
      VG_(printf)(" cr: %08x:%08x  xer: %08x:%08x\n",
		  tst->arch.m_cr, tst->arch.sh_cr,
		  tst->arch.m_xer, tst->arch.sh_xer);
   } else {
      for (i = 0; i < 32; ++i) {
	 if ((i & 7) == 0)
	    VG_(printf)("r%02d:", i);
	 VG_(printf)(" %08x", tst->arch.m_gpr[i]);
	 if ((i & 7) == 7)
	    VG_(printf)("\n");
      }
      VG_(printf)(" lr: %08x  ctr: %08x  cr: %08x  xer: %08x\n",
		  tst->arch.m_lr, tst->arch.m_ctr,
		  tst->arch.m_cr, tst->arch.m_xer);
   }
   VG_(printf)("NIP: %08x\n", tst->arch.m_eip);
}

void VG_(arch_process_options)(void)
{
}

void VG_(arch_clear_thread)( arch_thread_t *regs )
{
}

void VG_(arch_thread_init)( arch_thread_t *regs )
{
   /* Zero out the main thread's FP and altivec state. */
   VG_(memset)(&regs->m_fpr[0], 0, 33 * sizeof(double));
   VG_(baseBlock)[VGOFF_(fpu_state_ptr)] = (Addr) &regs->m_fpr[0];

   if (VG_(hardware_capabilities) & VKI_HWCAP_ALTIVEC) {
      VG_(memset)(&regs->m_vr[0], 0, 33 * 16);
      VG_(baseBlock)[VGOFF_(vec_state_ptr)] = (Addr) &regs->m_vr[0];
   }
}

void VG_(arch_thread_cleanup) ( arch_thread_t *regs )
{
}

void VG_(arch_child_setup) ( arch_thread_t *regs, arch_thread_t *parent_regs )
{
   memcpy(regs->m_fpr, parent_regs->m_fpr, sizeof(regs->m_fpr));
   memcpy(regs->m_vr, parent_regs->m_vr, sizeof(regs->m_vr));
}

Int VG_(arch_set_child_regs_from_BB)(Int pid)
{
   int r;

   for (r = 0; r < 32; ++r)
      if (ptrace(PTRACE_POKEUSER, pid, (PT_R0 + r) * 4, VG_(baseBlock)[r])) {
	 VG_(printf)("error setting gpr%d: %s\n", r, strerror(errno));
	 return -1;
      }
   if (ptrace(PTRACE_POKEUSER, pid, PT_NIP * 4, VG_(baseBlock)[VGOFF_(m_eip)]) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CCR * 4, VG_(baseBlock)[VGOFF_(m_cr)]) ||
       ptrace(PTRACE_POKEUSER, pid, PT_LNK * 4, VG_(baseBlock)[VGOFF_(m_lr)]) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CTR * 4, VG_(baseBlock)[VGOFF_(m_ctr)]) ||
       ptrace(PTRACE_POKEUSER, pid, PT_XER * 4, VG_(baseBlock)[VGOFF_(m_xer)]))
      return -1;
   return 0;
}

Int VG_(arch_set_child_regs)(Int pid, arch_thread_t *tst)
{
   int r;

   for (r = 0; r < 32; ++r)
      if (ptrace(PTRACE_POKEUSER, pid, (PT_R0 + r) * 4, tst->m_gpr[r])) {
	 VG_(printf)("error setting gpr%d: %s\n", r, strerror(errno));
	 return -1;
      }
   if (ptrace(PTRACE_POKEUSER, pid, PT_NIP * 4, tst->m_eip) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CCR * 4, tst->m_cr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_LNK * 4, tst->m_lr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_CTR * 4, tst->m_ctr) ||
       ptrace(PTRACE_POKEUSER, pid, PT_XER * 4, tst->m_xer))
      return -1;

   return 0;
}

UInt *VG_(arch_reg_addr_from_BB)(Int regno)
{
   Int idx;

   if (0 <= regno && regno <= 31)
      idx = VGOFF_(m_gpr) + regno;
   else
      switch (regno) {
      case R_CR:	idx = VGOFF_(m_cr);	break;
      case R_LR:	idx = VGOFF_(m_lr);	break;
      case R_CTR:	idx = VGOFF_(m_ctr);	break;
      case R_XER:	idx = VGOFF_(m_xer);	break;
      default:		return NULL;
      }
   return &VG_(baseBlock)[idx];
}


UInt *VG_(arch_reg_addr)(Int regno, arch_thread_t *tst)
{
   UInt *ret = 0;

   if (0 <= regno && regno <= 31)
      ret = &tst->m_gpr[regno];
   else
      switch (regno) {
      case R_CR:	ret = &tst->m_cr;	break;
      case R_LR:	ret = &tst->m_lr;	break;
      case R_CTR:	ret = &tst->m_ctr;	break;
      case R_XER:	ret = &tst->m_xer;	break;
      }

   return ret;
}

void arch_set_arg_and_ret(ThreadId tid, UInt arg, Addr ret)
{
   /* Set R3 to the argument and LR to the bogus return address. */
   SET_PTHREQ_ARG(tid, arg);
   SET_PTHREQ_RETADDR(tid, ret);
}

void arch_thread_initial_stack(ThreadId tid, UInt arg, Addr ret)
{
   UInt	    esp;

   esp = ARCH_STACK_POINTER(VG_(threads)[tid].arch);
   /* make sure top of stack is 16-byte aligned */
   esp &= ~0xf;
   /* make an initial stack frame and initialize the link to 0 */
   esp -= 16;
   SET_PTHREQ_ESP(tid, esp);

   VG_TRACK(new_mem_stack, esp, 16);
   VG_TRACK(pre_mem_write, Vg_CorePThread, tid, "new thread: stack", esp, 4);
   *(UInt *)esp = 0;
   VG_TRACK(post_mem_write, esp, 4);

   /* Set r3 to the argument and lr to the bogus return address */
   SET_PTHREQ_ARG(tid, arg);
   SET_PTHREQ_RETADDR(tid, ret);
}
