
/*--------------------------------------------------------------------*/
/*--- Low-level syscall code for x86.             vg_x86_syscall.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible x86 protected-mode
   emulator for monitoring program execution on x86-Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/


#include "vg_include.h"
#include "vg_proxylwp.h"

extern void do_thread_syscall(Int sys, 
			      Int arg1, Int arg2, Int arg3, Int arg4, Int arg5, Int arg6,
			      Int *result, enum PXState *statep, enum PXState poststate);

asm(
".text\n"
"	.type do_thread_syscall,@function\n"

"do_thread_syscall:\n"
"	push	%esi\n"
"	push	%edi\n"
"	push	%ebx\n"
"	push	%ebp\n"
".sys_before:\n"
"	movl	16+ 4(%esp),%eax\n" /* syscall */
"	movl	16+ 8(%esp),%ebx\n" /* arg1 */
"	movl	16+12(%esp),%ecx\n" /* arg2 */
"	movl	16+16(%esp),%edx\n" /* arg3 */
"	movl	16+20(%esp),%esi\n" /* arg4 */
"	movl	16+24(%esp),%edi\n" /* arg5 */
"	movl	16+28(%esp),%ebp\n" /* arg6 */
".sys_restarted:\n"
"	int	$0x80\n"
".sys_after:\n"
"	movl	16+32(%esp),%ebx\n"	/* ebx = Int *res */
"	movl	%eax, (%ebx)\n"		/* write the syscall retval */

"	movl	16+36(%esp),%ebx\n"	/* ebx = enum PXState * */
"	testl	%ebx, %ebx\n"
"	jz	1f\n"

"	movl	16+40(%esp),%ecx\n"	/* write the post state (must be after retval write) */
"	movl	%ecx,(%ebx)\n"

".sys_done:\n"				/* OK, all clear from here */
"1:	popl	%ebp\n"
"	popl	%ebx\n"
"	popl	%edi\n"
"	popl	%esi\n"
"	ret\n"
"	.size do_thread_syscall,.-do_thread_syscall\n"
".previous\n"

".section .rodata\n"
"	.globl	sys_before\n"
"sys_before:	.long	.sys_before\n"
"	.globl	sys_restarted\n"
"sys_restarted:	.long	.sys_restarted\n"
"	.globl	sys_after\n"
"sys_after:	.long	.sys_after\n"
"	.globl	sys_done\n"
"sys_done:	.long	.sys_done\n"
".previous\n"
);


/* Run a syscall for a particular thread, getting the arguments from
   the thread's registers, and returning the result in the thread's
   eax.

   Assumes that the only thread state which matters is the contents of
   %eax-%ebp and the return value in %eax.
 */
void VG_(thread_syscall)(Int syscallno, arch_thread_t *tst, 
			 enum PXState *state , enum PXState poststate)
{
   do_thread_syscall(syscallno,   /* syscall no. */
		     tst->m_ebx,  /* arg 1 */
		     tst->m_ecx,  /* arg 2 */
		     tst->m_edx,  /* arg 3 */
		     tst->m_esi,  /* arg 4 */
		     tst->m_edi,  /* arg 5 */
		     tst->m_ebp,  /* arg 6 */
		     &tst->m_eax, /* result */
		     state,	  /* state to update */
		     poststate);  /* state when syscall has finished */
}

/* Back up to restart a system call. */
void VG_(arch_restart_syscall)(arch_thread_t *tst)
{
   tst->m_eip -= 2;		/* sizeof(int $0x80) */

   /* Make sure our caller is actually sane, and we're really backing
      back over a syscall.

      int $0x80 == CD 80 
   */
   {
      UChar *p = (UChar *)tst->m_eip;
      
      if (p[0] != 0xcd || p[1] != 0x80)
	 VG_(message)(Vg_DebugMsg, 
		      "?! restarting over syscall at %p %02x %02x\n",
		      tst->m_eip, p[0], p[1]);

      vg_assert(p[0] == 0xcd && p[1] == 0x80);
   }
}
