/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/

#ifndef __KALC_H
#define __KALC_H 1

#ifdef __MSDOS__
#  define MSDOS 1
#endif

#include <stdio.h>
#include <setjmp.h>

#include "cmp.h"


/***********************
 * General definitions *
 ***********************/
#define COPYRIGHT "kalc v2.2.2, Copyright (C) 1999-2000 Eduardo M Kalinowski " \
"(ekalin@iname.com)\n\n"

#define HELPMSG \
"Usage: kalc [OPTION]... [COMMANDS]...\n"			\
"   or: kalc --version | --help\n\n"				\
"  -d <n>     Display at most <n> stack levels\n"		\
"  -s         Do not load stack from init file\n"		\
"  -m         Do not load memory from init file\n"		\
"  -b         Batch mode: exit after processing command line\n"	\
"  --help     Display help message and exit successfully\n"	\
"  --version  Display version and exit successfully\n"

#define MAGICVER    "kalc220"
#define MAGICVER210 "kalc210"
#define MAGICVER200 "calc200"


/* Change here if you want more stack levels */
#ifndef STACKINCREMENT
#  define STACKINCREMENT 25
#endif

/* Change here if you want more space for the prompt */
#define MAXPROMPTSIZE 20

/* Maximum path for status file -- should be more than enough */
#define PATHSIZE 200

/* Number of levels composite objects may nest */
#define COMPLEVELS 64

/* There is no need to save more arguments, for the maximum number of
   arguments a function takes currently is 3. */
#define LASTARGS 3

/**********
 * Errors *
 **********/
enum Errors { ERR_NOERR            = 0,
	      ERR_TOOFEWARGUMENTS  = 11,
	      ERR_BADARGUMENTTYPE  = 12,
	      ERR_BADARGUMENTVALUE = 13,
	      ERR_INVALIDDATE      = 14,
	      ERR_NONREALRESULT    = 21,
	      /* ERR_UNDEFINEDRESULT  = 22, */   /* Not used anymore */
	      ERR_STACKOVERFLOW    = 31,
	      ERR_NOTENOUGHMEMORY  = 32,
	      /* ERR_INVALIDNUMBER    = 41, */   /* Not used anymore */
	      ERR_SYNTAXERROR      = 42,
	      ERR_UNDEFINEDNAME    = 51,
	      ERR_COULDNOTOPENFILE = 61,
	      ERR_COULDNOTCHDIR    = 62,
	      ERR_INVALIDVERSION   = 63,
	      ERR_MISSINGOBJECT    = 71,
	      ERR_UNFINISHEDCOMP   = 72,
	      ERR_UNSTARTEDPROG    = 73,
	      ERR_MAXLOCALENVS     = 81,
	      ERR_NOTEMPENVS       = 82,
	      ERR_NOTINPROGRAM     = 83,
	      ERR_RECURSIONTOODEEP = 84,
	      ERR_CUSTOM           = 9999,
};


/**************************
 * Stack object structure *
 **************************/
typedef struct tagTagged
{
  char *tag;
  struct tagObject *obj;
} Tagged;

typedef struct tagComposite
{
  struct tagObject *obj;
  struct tagComposite *prev;
  struct tagComposite *next;
} Composite;

typedef struct tagObject
{
  int type;
  union {
    double real;
    Complex cmp;
    hxs h;
    struct tagTagged tagged;
    struct tagComposite *comp;
    unsigned char *str;
    void (*func)(void);
  } value;
} Object;


/*********
 * Types *
 *********/
enum Types { TYPE_REAL        = 0,
	     TYPE_CMP         = 1,
	     TYPE_STR         = 2,
	     TYPE_ID          = 6,
	     TYPE_PROG        = 8,
	     TYPE_HXS         = 10,
	     TYPE_TAGGED      = 12,
	     TYPE_FUNC        = 18,
	     TYPE_UNQUOTEDID  = 50,
	     TYPE_NOTANOBJECT = -1,
};

#define type(obj) ((obj).type)


/***********************************
 * Structure with the user options *
 ***********************************/
typedef struct tagOptions
{
  unsigned angleMode:     1;
  unsigned coordinate:    2;
  unsigned numberFormat:  2;
  unsigned decimalPlaces: 4;
  unsigned expert:        1;
  unsigned base:          6;
  unsigned wordSize;
  int nLines;
  int width;
  char prompt[MAXPROMPTSIZE + 1];
} Options;

/* Angle modes */
enum AngleModes { ANGLE_RAD = 0,
		  ANGLE_DEG,
};

/* Coordinate modes */
enum CoordinateModes { COORD_RECT  = 0,
		       COORD_CYLIN,
};

/* Number modes */
enum NumberFormats { FMT_STD = 0,
		     FMT_FIX,
		     FMT_SCI,
		     FMT_ENG,
};


/*************************************************************
 * Structure for the parser -- functions and their addresses *
 *************************************************************/
typedef struct tagFunctionList {
  char *name;
  void (*address)(void);
} FunctionList;


/********************************************************
 * Used by program execution of control flow statements *
 ********************************************************/
typedef enum tagCtrFlowType {
  CTR_IF, CTR_CASE, CTR_DEFLOOP, CTR_WHILE, CTR_DO
} CtrFlowType;


typedef struct tagCtrFlow 
{
  CtrFlowType type;
  Composite *address;
  double currValue;
  double endValue;
  char *ctrVarName;
} CtrFlow;


/********************
 * Global variables *
 ********************/
extern Object **s;                /* The stack */
extern Object **tos;              /* Pointer to top of stack */
extern int      currStackCap;     /* Current stack capacity */
extern jmp_buf  jmpError;         /* Location to jump in case of errors */
extern int      linesToDisplay;   /* Stack lines to display */
extern Options  userOptions;      /* Options the user set */
extern int      lastArgN;         /* Number of arguments last command took */
extern Object  *lastArgs[];       /* Pointers to the last arguments */
extern char     currStatusFile[]; /* Path of currently "open" status file */
extern int      dirty;            /* Has the status file been modified? */
extern int      stackMarks[];     /* Used by composite object parsing */
extern int      nMarks;           /* Current number of marks */
extern int      progNesting;      /* Current nesting level of programs */
extern int      lastArrowLevel[]; /* Levels at which -> was found */
extern int      lastArrowDepth[]; /* Correspoding depth */
extern int      nArrows;          /* Marks lastArrow stack level */
extern int      currMemLevel;     /* Current memory stack level */
extern int      nFlow;            /* Marks flow structure stack depth */

/*********************
 * General functions *
 *********************/
#define enoughArgs(n) (!(tos - s < (n)))
/* #define getPadding() ((int)re_xpon(MAXSTACKITEMS) + 1) */
#define MAXWORDSIZE (sizeof(hxs) << 3)
#define canRead(ver) ((strcmp(ver, MAGICVER) == 0) || \
		      (strcmp(ver, MAGICVER210) == 0))

void readCommands(void);
void showStack(int n);
void initialize(int loadStack, int loadMem);
void finalize(void);
void setDefaults(void);
int insertStackElement(int count, int key);
void freeObjectSpecials(Object obj);


/****************************************
 * Status file loading/saving functions *
 ****************************************/
#define readUserOptions(f) fread(&userOptions, sizeof(userOptions), 1, f); \
                           linesToDisplay = userOptions.nLines
#define storeUserOptions(f) (fwrite(&userOptions, sizeof(userOptions), 1, f))

int wantsToContinue(void);
int readStatusFromFile(char *file, int loadStack, int loadMem);
int readStatus(FILE *fp, int loadStack, int loadMem);
int saveStatusToFile(char *file);
void saveStatus(FILE *fp);
void updateCurrentStatusFile(char *file);
void saveStack(FILE *fp);
void saveObject(Object *obj, FILE *fp);
Object loadObject(FILE *fp);
void saveString(const char *str, FILE *fp);
char *loadString(FILE *fp);
void saveProgram(Composite *comp, FILE *fp);
Composite *loadProgram(FILE *fp);


/*************************
 * Version 2.0.0 support *
 *************************/
int readStatus200(FILE *fp, int loadStack, int loadMem);
Object loadObject200(FILE *fp);
void loadMemory200(FILE *fp);


/********************
 * Parser functions *
 ********************/
#define ltrim(str) while (isspace(*(str))) \
                     (str)++

void parseLine(char *l);
Object processToken(char *tok);
void evaluateObj(Object obj, int evalIDs);
void initializeTokenizer(char *line);
char *getNextToken(void);
char *getTillEOL(void);
int parseReal(char *str, double *real);
int parseComplex(char *str, Complex *cmp);
int parseHxs(char *str, hxs *h);


/*********************
 * Command execution *
 *********************/
#define freeSpace() (tos - s != currStackCap)
void run1_1_Function(Object (*f)(Object n, int *err), char *name);
void run2_1_Function(Object (*f)(Object n, Object p, int *err), char *name);
void run3_1_Function(Object (*f)(Object n, Object p, Object q, int *err), \
		     char *name);
void run1_2_Function(Object (*f)(Object n, Object *r, int *err), char *name);
void run1_0_Function(int (*f)(Object n), char *name);
#if 0
void run2_0_Function(int (*f)(Object n, Object p), char *name);
#endif
void saveLastArgs(int n);
void freeLastArgs(void);
void storeLastArgs(FILE *fp);
void loadLastArgs(FILE *fp);
void doError(char *name, int errCode);
/*
  The weird thing below is a function that takes a pointer to
  character as argument, and returns a pointer to a function that
  takes no arguments and returns nothing.
*/
void (*getFuncAddress(char *name))(void);
char *getFuncName(void (*addr)(void));


/************************
 * Decompiler functions *
 ************************/
#define stackDecomp(n) (decomp(n,                              \
			       (int) userOptions.numberFormat, \
			       (int) userOptions.decimalPlaces))
#define editDecomp(n) (decomp(n, FMT_STD, 0))
char *decomp(Object n, int numberFormat, int precision);
char *stackPrepare(Object n, int level);
int needsWrapping(char *str, int maxlen);
char *wordWrap(char *str, int maxlen, int padding);
char *getNextWord(char *str);
char *realDecomp(double n, int numberFormat, int precision);
char *cmpDecomp(Complex n, int numberFormat, int precision);
char *hxsDecomp(hxs h);
char forDigit(int n, int base);
char *reverseStr(char *str, int start);
char *progDecomp(Composite *comp);


/**********************************************
 * User commands for dealing with status file *
 **********************************************/
void f_save(void);
void f_saveAs(void);
void f_open(void);
void f_load(void);
void f_coldStart(void);


/*******************************
 * Stack manipulation commands *
 *******************************/
Object *objdup(Object *obj);
void insertObject(Object obj);
void insertNoDup(Object obj);
void insertReal(double d);
void insertId(char *str);

void f_drop(void);
void f_nip(void);
void f_drop2(void);
void f_clear(void);
void f_rot(void);
void f_unrot(void);
void f_dup(void);
void f_dupdup(void);
void f_ndupn(void);
void f_dup2(void);
void f_dupn(void);
void f_swap(void);
void f_dropn(void);
void f_over(void);
void f_pick3(void);
void f_pick(void);
void f_unpick(void);
void f_roll(void);
void f_rolld(void);
void f_keep(void);
void f_depth(void);

void _f_drop(void);
void _f_dropn(int n, char *name);
void _f_swap(void);
void _f_roll(int n, char *name);
void _f_rolld(int n, char *name);
#define _f_depth() (tos - s)


/***********************
 * Arithmetic commands *
 ***********************/
void f_subtract(void);
void f_multiply(void);
void f_divide(void);
void f_inv(void);
void f_abs(void);
void f_ceil(void);
void f_floor(void);
void f_pctOf(void);
void f_pctT(void);
void f_pctCh(void);
void f_mod(void);
void f_gcd(void);
void f_lcm(void);

Object _f_subtract(Object n, Object p, int *err);
Object _f_multiply(Object n, Object p, int *err);
Object _f_divide(Object n, Object p, int *err);
Object _f_inv(Object n, int *err);
Object _f_abs(Object n, int *err);
Object _f_ceil(Object n, int *err);
Object _f_floor(Object n, int *err);
Object _f_pctOf(Object n, Object p, int *err);
Object _f_pctT(Object n, Object p, int *err);
Object _f_pctCh(Object n, Object p, int *err);
Object _f_mod(Object n, Object p, int *err);
Object _f_gcd(Object n, Object p, int *err);
Object _f_lcm(Object n, Object p, int *err);


/************************
 * Exponential commands *
 ************************/
void f_exp(void);
void f_ln(void);
void f_pow(void);
void f_xroot(void);
void f_sq(void);
void f_sqrt(void);
void f_expm1(void);
void f_lnp1(void);
void f_log(void);
void f_alog(void);
void f_cis(void);

Object _f_exp(Object n, int *err);
Object _f_ln(Object n, int *err);
Object _f_pow(Object base, Object exp, int *err);
Object _f_xroot(Object n, Object p, int *err);
Object _f_sq(Object n, int *err);
Object _f_sqrt(Object n, int *err);
Object _f_expm1(Object n, int *err);
Object _f_lnp1(Object n, int *err);
Object _f_log(Object n, int *err);
Object _f_alog(Object n, int *err);
Object _f_cis(Object n, int *err);


/**************************
 * Trigonometric commands *
 **************************/
#undef PI
#define PI 3.14159265358979323846264338327

void f_pi(void);
void f_sin(void);
void f_cos(void);
void f_tan(void);
void f_sec(void);
void f_csc(void);
void f_cot(void);
void f_asin(void);
void f_acos(void);
void f_atan(void);
void f_atan2(void);
void f_asec(void);
void f_acsc(void);
void f_acot(void);
void f_vers(void);
void f_hav(void);
void f_dTOr(void);
void f_rTOd(void);

Object _f_sin(Object n, int *err);
Object _f_cos(Object n, int *err);
Object _f_tan(Object n, int *err);
Object _f_sec(Object n, int *err);
Object _f_csc(Object n, int *err);
Object _f_cot(Object n, int *err);
Object _f_asin(Object n, int *err);
Object _f_acos(Object n, int *err);
Object _f_atan(Object n, int *err);
Object _f_atan2(Object n, Object p, int *err);
Object _f_asec(Object n, int *err);
Object _f_acsc(Object n, int *err);
Object _f_acot(Object n, int *err);
Object _f_vers(Object n, int *err);
Object _f_hav(Object n, int *err);
Object _f_dTOr(Object n, int *err);
Object _f_rTOd(Object n, int *err);


/***********************
 * Hyperbolic commands *
 ***********************/
void f_sinh(void);
void f_cosh(void);
void f_tanh(void);
void f_asinh(void);
void f_acosh(void);
void f_atanh(void);
void f_sech(void);
void f_csch(void);
void f_coth(void);
void f_asech(void);
void f_acsch(void);
void f_acoth(void);
void f_gd(void);
void f_invgd(void);

Object _f_sinh(Object n, int *err);
Object _f_cosh(Object n, int *err);
Object _f_tanh(Object n, int *err);
Object _f_asinh(Object n, int *err);
Object _f_acosh(Object n, int *err);
Object _f_atanh(Object n, int *err);
Object _f_sech(Object n, int *err);
Object _f_csch(Object n, int *err);
Object _f_coth(Object n, int *err);
Object _f_asech(Object n, int *err);
Object _f_acsch(Object n, int *err);
Object _f_acoth(Object n, int *err);
Object _f_gd(Object n, int *err);
Object _f_invgd(Object n, int *err);


/**********************************************
 * Complex number functions (conversion, etc) *
 **********************************************/
void f_rTOc(void);
void f_reTOc(void);
void f_imTOc(void);
void f_cTOr(void);
void f_re(void);
void f_im(void);
void f_conj(void);
void f_arg(void);

Object _f_rTOc(Object n, Object p, int *err);
Object _f_reTOc(Object n, int *err);
Object _f_imTOc(Object n, int *err);
Object _f_cTOr(Object n, Object *r, int *err);
Object _f_re(Object n, int *err);
Object _f_im(Object n, int *err);
Object _f_conj(Object n, int *err);
Object _f_arg(Object n, int *err);

void __f_reTOc(Object *n);


/*******************************
 * Hexadecimal string commands *
 *******************************/
#define hxs_add(h1, h2) (normalize((h1)+(h2)))
#define hxs_sub(h1, h2) (normalize((h1)-(h2)))
#define hxs_mul(h1, h2) (normalize((h1)*(h2)))
#define hxs_div(h1, h2) ((h1 && h2) ? (normalize((h1)/(h2))) : 0)

hxs normalize(hxs h);

void f_rTOb(void);
void f_bTOr(void);
void f_sl(void);
void f_slb(void);
void f_sr(void);
void f_srb(void);
void f_rl(void);
void f_rlb(void);
void f_rr(void);
void f_rrb(void);
void f_asr(void);

Object _f_rTOb(Object n, int *err);
Object _f_bTOr(Object n, int *err);
Object _f_sl(Object n, int *err);
Object _f_slb(Object n, int *err);
Object _f_sr(Object n, int *err);
Object _f_srb(Object n, int *err);
Object _f_rl(Object n, int *err);
Object _f_rlb(Object n, int *err);
Object _f_rr(Object n, int *err);
Object _f_rrb(Object n, int *err);
Object _f_asr(Object n, int *err);

void __f_rTOb(Object *n);


/***********************
 * Relational commands *
 ***********************/
void f_eq(void);
void f_neq(void);
void f_lt(void);
void f_gt(void);
void f_lte(void);
void f_gte(void);

Object _f_eq(Object n, Object p, int *err);
Object _f_neq(Object n, Object p, int *err);
Object _f_lt(Object n, Object p, int *err);
Object _f_gt(Object n, Object p, int *err);
Object _f_lte(Object n, Object p, int *err);
Object _f_gte(Object n, Object p, int *err);


/**************************
 * Miscellaneous commands *
 **************************/
void exitAbort(void);
void f_ip(void);
void f_fp(void);
void f_rand(void);
void f_rdz(void);
void f_gamma(void);
void f_lgamma(void);
void f_perm(void);
void f_comb(void);
void f_max(void);
void f_min(void);
void f_sign(void);
void f_psign(void);
void f_mant(void);
void f_xpon(void);
void f_rnd(void);
void f_type(void);
void f_vtype(void);
void doShell(void);
void f_nan(void);
void f_inf(void);
void f_minusInf(void);
void f_eval(void);
void f_lastarg(void);

Object _f_ip(Object n, int *err);
Object _f_fp(Object n, int *err);
int _f_rdz(Object n);
Object _f_gammap1(Object n, int *err);
Object _f_lgamma(Object n, int *err);
Object _f_perm(Object n, Object p, int *err);
Object _f_comb(Object n, Object p, int *err);
Object _f_min(Object n, Object p, int *err);
Object _f_max(Object n, Object p, int *err);
Object _f_sign(Object n, int *err);
Object _f_psign(Object n, int *err);
Object _f_mant(Object n, int *err);
Object _f_xpon(Object n, int *err);
Object _f_rnd(Object n, Object p, int *err);
Object _f_type(Object n, int *err);
Object _f_vtype(Object n, int *err);


/**************************
 * Time and date commands *
 **************************/
void f_date(void);
void f_time(void);
void f_TOhms(void);
void f_hmsTO(void);
void f_dateAdd(void);
void f_ddays(void);
void f_hmsAdd(void);
void f_hmsSub(void);
void f_dow(void);
void f_dowstr(void);
void f_tstr(void);


/*******************
 * String commands *
 *******************/
void f_size(void);
void f_TOstr(void);
void f_strTO(void);
void f_num(void);
void f_chr(void);
void f_head(void);
void f_tail(void);
void f_pos(void);
void f_sub(void);
void f_repl(void);
void f_strTOid(void);
void f_idTOstr(void);

Object _f_strcat(Object n, Object p, int *err);
Object _f_size(Object n, int *err);
Object _f_TOstr(Object n, int *err);
Object _f_num(Object n, int *err);
Object _f_chr(Object n, int *err);
Object _f_pos(Object n, Object p, int *err);
Object _f_sub(Object n, Object p, Object q, int *err);
Object _f_repl(Object n, Object p, Object q, int *err);


/***************************
 * Tagged objects commands *
 ***************************/
void f_dtag(void);
void f_TOtag(void);

void dtag(Object **n);


/*************************************
 * Customization commands (and help) *
 *************************************/
void doSet(void);
void doShow(void);
void doHelp(void);
void showDirective(char *val);
void showHelp(char *topic);
void processDirective(char *dir, char *value);

void f_rad(void);
void f_deg(void);
void f_rect(void);
void f_cylin(void);
void f_std(void);
void f_fix(void);
void f_sci(void);
void f_eng(void);
void f_stws(void);
void f_rcws(void);
void f_bin(void);
void f_oct(void);
void f_dec(void);
void f_hex(void);

int _f_fix(Object n);
int _f_sci(Object n);
int _f_eng(Object n);
int _f_stws(Object n);


/*******************
 * Memory commands *
 *******************/
void doUnquoted(char *id);
void doSto(void);
void diskSto(void);
void doRcl(void);
void diskRcl(void);
void doPurge(void);
void doVars(void);
void doClvar(void);
void _doClvar(void);
void clearMemoryLevel(int level);
void doPwd(void);
void doCd(void);
void saveMemory(FILE *fp);
void loadMemory(FILE *fp);
void initNextMemoryLevel(void);
void bindObject(char *name, Object *objp);


/************************
 * Polymorphic commands *
 ************************/
void f_add(void);
void f_and(void);
void f_or(void);
void f_xor(void);
void f_not(void);
void f_neg(void);

Object _f_add(Object n, Object p, int *err);
Object _f_and(Object n, Object p, int *err);
Object _f_or(Object n, Object p, int *err);
Object _f_xor(Object n, Object p, int *err);
Object _f_not(Object n, int *err);
Object _f_neg(Object n, int *err);


/***********************************
 * Commands related to programming *
 ***********************************/
void startProgram(void);
void endProgram(void);
void evalProgram(Composite *comp);
void skipProgram(Composite **comp, void (*func1)(void), void (*func2)(void));
void skipToEndOfCase(Composite **comp);
void f_ift(void);
void f_ifte(void);
void f_dobind(void);
void f_abnd(void);
void f_arrow(void);
void f_if(void);
void f_then(void);
void f_else(void);
void f_end(void);
void f_case(void);
void f_start(void);
void f_for(void);
void f_next(void);
void f_step(void);
void incrLoop(double n);
void f_while(void);
void f_repeat(void);
void f_do(void);
void f_until(void);


#endif /* __KALC_H */
